<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\custom_error_message;

use app\components\validators\DataValidator;
use app\components\validators\DateValidator;
use app\components\validators\EmailValidator;
use app\components\validators\FileValidator;
use app\components\validators\MultipleEmailValidator;
use app\components\validators\NumberValidator;
use app\components\validators\PhoneValidator;
use app\components\validators\RecaptchaValidator;
use app\components\validators\RegularExpressionValidator;
use app\components\validators\RequiredValidator;
use app\components\validators\StringValidator;
use app\components\validators\UrlValidator;
use app\controllers\AjaxController;
use app\events\SubmissionErrorEvent;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormData;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\custom_error_message\models\CustomErrorMessage;
use app\modules\addons\modules\custom_error_message\models\CustomErrorMessageItem;

/**
 * Class Module
 * @package app\modules\addons\modules\custom_error_message
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{
    public $id = "custom_error_message";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'CustomErrorMessage' => CustomErrorMessage::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        $onValidatorErrorAdded = [
            SubmissionErrorEvent::EVENT_NAME => [
                [self::class, 'onValidatorErrorAdded']
            ]
        ];

        return [
            'app\models\Form' => [
                'beforeDelete' => [
                    ['app\modules\addons\modules\custom_error_message\Module', 'beforeDeleteAForm']
                ]
            ],
            DataValidator::class => $onValidatorErrorAdded,
            DateValidator::class => $onValidatorErrorAdded,
            EmailValidator::class => $onValidatorErrorAdded,
            FileValidator::class => $onValidatorErrorAdded,
            MultipleEmailValidator::class => $onValidatorErrorAdded,
            NumberValidator::class => $onValidatorErrorAdded,
            PhoneValidator::class => $onValidatorErrorAdded,
            RecaptchaValidator::class => $onValidatorErrorAdded,
            RegularExpressionValidator::class => $onValidatorErrorAdded,
            RequiredValidator::class => $onValidatorErrorAdded,
            StringValidator::class => $onValidatorErrorAdded,
            UrlValidator::class => $onValidatorErrorAdded,
        ];
    }

    public static function onValidatorErrorAdded($event)
    {
        if (isset($event->fieldName, $event->sender->submissionModel) && $event->sender->submissionModel instanceof FormSubmission) {
            $fieldName = $event->fieldName;
            $submissionModel = $event->sender->submissionModel;
            $model = CustomErrorMessage::findOne(['form_id' => $submissionModel->form_id, 'status' => 1]);

            if (isset($model, $model->form_id)) {
                /** @var FormData $dataModel */
                $dataModel = $submissionModel->form->formData;
                $fieldsForEmail = $dataModel->getFieldsForEmail();
                $submissionData = $submissionModel->getSubmissionData();
                $fieldValues = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);;
                foreach ($model->items as $item) {
                    if ($item->field_id === $fieldName) {
                        $message = $event->message;
                        $validationType = $item->field_validation_type;
                        $customErrorMessage = $item->field_validation_message;
                        $customErrorMessage = SubmissionHelper::replaceTokens($customErrorMessage, $fieldValues);
                        if (!empty($message)
                            && !empty($validationType)
                            && !empty($model->validationMessages)
                            && !empty($model->validationMessages[$validationType])
                            && strpos($message, $model->validationMessages[$validationType]) !== false) {
                            $event->message = $customErrorMessage;
                        }
                    }
                }
            }
        }
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = CustomErrorMessage::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new CustomErrorMessage();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new CustomErrorMessageItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * @param $event
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public static function beforeDeleteAForm($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            /** @var \app\modules\addons\modules\custom_error_message\models\CustomErrorMessage $model */
            $model = CustomErrorMessage::findOne(['form_id' => $event->sender->id]);
            if ($model) {
                $model->delete();
            }
        }
    }
}