<?php

namespace app\modules\addons\modules\constant_contact\services;

use Exception;
use Yii;

class ConstantContactService
{

    /** @string Base Url */
    public $baseUrl = "https://api.cc.email/v3";

    /** @string Token Url */
    public $tokenUrl = "https://authz.constantcontact.com/oauth2/default/v1/token";

    /** @var ConstantContactRestApi cr */
    protected $api;

    /** @string Client ID */
    protected $clientId;

    /** @string Client Secret */
    protected $clientSecret;

    /** @string Access Token */
    protected $accessToken;

    /** @string Access Token */
    protected $scope;

    /** @string Authorization */
    protected $authorization;

    protected $redirectUrl;

    /**
     * CleverReachService constructor.
     * @param $clientID
     * @param $clientSecret
     */
    public function __construct($clientID, $clientSecret, $redirectUrl)
    {
        $this->clientId = $clientID;
        $this->clientSecret = $clientSecret;
        $this->redirectUrl = $redirectUrl;
        $this->api = new ConstantContactRestApi($this->baseUrl);
    }

    /**
     * This function can be used to generate the URL an account owner would use to allow your app to access their account.
     * After visiting the URL, the account owner is prompted to log in and allow your app to access their account.
     * They are then redirected to your redirect URL with the authorization code appended as a query parameter. e.g.:
     * http://localhost:8888/?code={authorization_code}
     *
     * @param string - Redirect Url
     * @return string - Full Authorization URL
     */
    public function getAuthorizationURL($redirectUrl = null) {
        // Create authorization URL
        $baseURL = "https://authz.constantcontact.com/oauth2/default/v1/authorize";
        $redirectUrl = is_null($redirectUrl) ? $this->redirectUrl : $redirectUrl;
        $authURL = $baseURL . "?client_id="
            . $this->clientId . "&scope=contact_data+offline_access&response_type=code"
            . "&redirect_uri=" . urlencode($redirectUrl)
            . "&state=" . substr(str_shuffle(MD5(microtime())), 0, 14);

        return $authURL;
    }

    /**
     * This function can be used to exchange an authorization code for an access token.
     * Make this call by passing in the code present when the account owner is redirected back to you.
     * The response will contain an 'access_token' and 'refresh_token'
     *
     * @param $redirectURI - URL Encoded Redirect URI
     * @param $clientId - API Key
     * @param $clientSecret - API Secret
     * @param $code - Authorization Code
     * @return string - JSON String of results
     */
    public function getAccessToken($redirectURI, $code) {
        // Use cURL to get access token and refresh token
        $ch = curl_init();

        // Create full request URL
        $url = $this->tokenUrl . '?code=' . $code . '&redirect_uri=' . urlencode($redirectURI) . '&grant_type=authorization_code&scope=contact_data';
        curl_setopt($ch, CURLOPT_URL, $url);

        // Set authorization header
        // Make string of "API_KEY:SECRET"
        $auth = $this->clientId . ':' . $this->clientSecret;
        // Base64 encode it
        $credentials = base64_encode($auth);
        // Create and set the Authorization header to use the encoded credentials
        $authorization = 'Authorization: Basic ' . $credentials;
        curl_setopt($ch, CURLOPT_HTTPHEADER, array($authorization));

        // Set method and to expect response
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        // Make the call
        $result = curl_exec($ch);
        curl_close($ch);
        return $result;
    }

    /***
     * @param $refreshToken - The refresh token provided with the previous access token
     * @return string - JSON String of results
     */
    public function refreshToken($refreshToken) {
        // Use cURL to get a new access token and refresh token
        $ch = curl_init();

        // Create full request URL
        $url = $this->tokenUrl . '?refresh_token=' . $refreshToken . '&grant_type=refresh_token';
        curl_setopt($ch, CURLOPT_URL, $url);

        // Set authorization header
        // Make string of "API_KEY:SECRET"
        $auth = $this->clientId . ':' . $this->clientSecret;
        // Base64 encode it
        $credentials = base64_encode($auth);
        // Create and set the Authorization header to use the encoded credentials
        $authorization = 'Authorization: Basic ' . $credentials;
        curl_setopt($ch, CURLOPT_HTTPHEADER, array($authorization));

        // Set method and to expect response
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        // Make the call
        $result = curl_exec($ch);
        curl_close($ch);
        return $result;
    }

    /**
     * @param $accessToken
     */
    public function setToken($accessToken)
    {
        $this->accessToken = $accessToken;
    }

    /**
     * @return mixed
     */
    public function getToken()
    {
        return $this->accessToken;
    }

    /**
     * Get Subscriber Lists
     *
     * @return array
     * @throws Exception
     */
    public function getSubscriberLists()
    {
        $this->api->setAuthMode("bearer", $this->accessToken);
        $response  =  (array) $this->api->get("/contact_lists");
        
        $lists = [];
        if (!empty($response) && !empty($response['lists']) && is_array($response['lists'])) {
            foreach($response['lists'] as $list) {
                $list = (array) $list;
                if (isset($list['list_id'], $list['name'])) {
                    $lists[$list['list_id']] = $list['name'];
                }
            }
        }

        return $lists;
    }

    /**
     * Get Fields
     *
     * @return array
     */
    public function getFields()
    {
        return [
            'first_name' => Yii::t('app', 'First Name'),
            'last_name' => Yii::t('app', 'Last Name'),
            'prefix_name' => Yii::t('app', 'Prefix Name'),
            'job_title' => Yii::t('app', 'Job Title'),
            // 'confirmed' => Yii::t('app', 'Confirmed'),
            'source' => Yii::t('app', 'Source'),
            'company_name' => Yii::t('app', 'Company Name'),
            'home_phone' => Yii::t('app', 'Home Phone'),
            'work_phone' => Yii::t('app', 'Work Phone'),
            'cell_phone' => Yii::t('app', 'Cell Phone'),
            'fax' => Yii::t('app', 'Fax'),
            'notes' => Yii::t('app', 'Notes'),
        ];
    }

    /**
     * Get Custom Fields
     *
     * @return array
     * @throws Exception
     */
    public function getCustomFields()
    {
        $this->api->setAuthMode("bearer", $this->accessToken);
        $response  =  (array) $this->api->get("/contact_custom_fields");

        $fields = [];

        if (!empty($response) && !empty($response['custom_fields']) && is_array($response['custom_fields'])) {
            foreach($response['custom_fields'] as $customField) {
                $customField = (array) $customField;
                if (isset($customField['custom_field_id'], $customField['label'])) {
                    $fields[$customField['custom_field_id']] = $customField['label'];
                }
            }
        }

        return $fields;
    }

    /**
     * Add or Edit Subscriber
     *
     * @param $contact
     * @return array|false
     * @throws Exception
     */
    public function syncContact($contact)
    {
        $this->api->setAuthMode("bearer", $this->accessToken);
        return $this->api->post("/contacts/sign_up_form", $contact);
    }
}