<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\constant_contact\controllers;

use app\components\User;
use app\helpers\ArrayHelper;
use app\models\Form;
use app\modules\addons\modules\constant_contact\models\ConstantContact;
use app\modules\addons\modules\constant_contact\models\ConstantContactField;
use app\modules\addons\modules\constant_contact\models\ConstantContactSearch;
use app\modules\addons\modules\constant_contact\services\ConstantContactException;
use app\modules\addons\modules\constant_contact\services\ConstantContactService;
use Exception;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\helpers\Json;
use yii\helpers\Url;
use yii\web\Controller;
use yii\web\NotFoundHttpException;

/**
 * AdminController implements the CRUD actions for ConstantContact model.
 */
class AdminController extends Controller
{

    /** @var ConstantContactService $ctctService */
    public $ctctService;
    public $redirectUrl;

    /**
     * @inheritdoc
     */
    public function init()
    {
        /** @var \app\modules\addons\modules\constant_contact\Module $module */
        $module = $this->module;
        $this->redirectUrl = $module->redirectUrl;
        $this->ctctService = new ConstantContactService($module->apiKey, $module->consumerSecret, $this->redirectUrl);

        parent::init();
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['POST'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create', 'check'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('ConstantContact')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = ConstantContact::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @return array
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\constant_contact\models\ConstantContact',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all ConstantContact models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new ConstantContactSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single ConstantContact model.
     * @param integer $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Check Constant Contact code to generate the Access Token
     *
     * @return \yii\web\Response
     */
    public function actionCheck()
    {
        $request = Yii::$app->request;

        $accessToken = $this->ctctService->getAccessToken($this->redirectUrl, $request->get('code'));
        $token = Json::decode($accessToken, true);

        if (isset($token['access_token'], $token['refresh_token'], $token['token_type'])) {

            $session = Yii::$app->session;
            $session->set('access_token', $token['access_token']);
            $session->set('refresh_token', $token['refresh_token']);
            $session->set('token_type', $token['token_type']);

            if ($session->has('redirectTo')) {
                $redirectTo = $session->get('redirectTo');
                $session->remove('redirectTo');
                header("Location: $redirectTo");
                exit();
            }

            $redirectTo = Url::to(['/addons/constant_contact/admin/create'], true);
            header("Location: $redirectTo");
            exit();
        }

        Yii::$app->getSession()->setFlash(
            'danger',
            Yii::t('app', 'Constant Contact authentication error.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Creates a new ConstantContact model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {

        $lists = [];
        $fields = [];

        $request = Yii::$app->request;
        $session = Yii::$app->session;

        $model = new ConstantContact();

        try {

            if ($request->isGet && !$session->has('access_token')) {

                $authorizationURL = $this->ctctService->getAuthorizationUrl();
                header("Location: $authorizationURL");
                exit();

            } elseif ($request->isGet && $session->has('access_token')) {

                $this->ctctService->setToken($session->get('access_token'));
                $lists = $this->ctctService->getSubscriberLists();
                $fields = array_merge($this->ctctService->getFields(), $this->ctctService->getCustomFields());

                $model->access_token = Json::encode([
                    'access_token' => $session->get('access_token'),
                    'refresh_token' => $session->get('refresh_token'),
                    'token_type' => $session->get('token_type'),
                    'unix_timestamp' => time(),
                ]);

                $session->remove('access_token');
                $session->remove('refresh_token');
                $session->remove('token_type');

            } elseif ($request->isPost && $model->load(Yii::$app->request->post())) {

                /**
                 * Save Constant Contact model and related items
                 */

                if ($model->validate()) {
                    $model->items = Yii::$app->request->post('ConstantContactField',[]);
                    $model->save(false);

                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The sign up form has been successfully configured.')
                    );

                    return $this->redirect(['index']);

                } else {
                    Yii::$app->getSession()->setFlash(
                        'danger',
                        Yii::t('app', 'An unexpected error has occurred. Please try again.')
                    );
                    $lists = $this->ctctService->getSubscriberLists();
                    $fields = array_merge($this->ctctService->getFields(), $this->ctctService->getCustomFields());
                }
            }

        } catch (\Exception $e) {
            // Show error message
            Yii::$app->getSession()->setFlash(
                'danger',
                $e->getMessage()
            );
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'fieldModel' => new ConstantContactField,
            'forms' => $forms,
            'lists' => $lists,
            'fields' => $fields,
        ]);
    }

    /**
     * Updates an existing ConstantContact model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $session = Yii::$app->session;

        $lists = [];
        $fields = [];

        $request = Yii::$app->request;

        if ($model->load($request->post()) && $model->validate()) {

            $model->items = Yii::$app->request->post('ConstantContactField',[]);
            $model->save();

            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The sign up form has been successfully updated.')
            );

            return $this->redirect(['index']);

        } else {

            $token = Json::decode($model->access_token, true);

            try {

                // Refresh access token
                if (isset($token['access_token'], $token['refresh_token'], $token['token_type'])) {
                    $token = Json::decode($model->access_token, true);
                    $accessToken = $this->ctctService->refreshToken($token['refresh_token']);
                    $token = Json::decode($accessToken, true);
                }

                if (isset($token['error'], $token['invalid_grant'])) {
                    // Request new access token to the user
                    $session->set('redirectTo', Url::current());
                    $authorizationURL = $this->ctctService->getAuthorizationUrl();
                    header("Location: $authorizationURL");
                    exit();
                }

                if (isset($token['access_token'], $token['refresh_token'], $token['token_type'])) { // If access token was saved on DB

                    $this->ctctService->setToken($token['access_token']);
                    $lists = $this->ctctService->getSubscriberLists();
                    $fields = array_merge($this->ctctService->getFields(), $this->ctctService->getCustomFields());

                } elseif ($session->has('access_token')) { // If access token was saved on Session

                    $this->ctctService->setToken($session->get('access_token'));
                    $lists = $this->ctctService->getSubscriberLists();
                    $fields = array_merge($this->ctctService->getFields(), $this->ctctService->getCustomFields());

                    // Save to DB
                    $model->access_token = Json::encode([
                        'access_token' => $session->get('access_token'),
                        'refresh_token' => $session->get('refresh_token'),
                        'token_type' => $session->get('token_type'),
                        'unix_timestamp' => time(),
                    ]);
                    $model->save();

                    // Delete session data
                    $session->remove('access_token');
                    $session->remove('refresh_token');
                    $session->remove('token_type');

                } else { // Else, request new access token to the user

                    $session->set('redirectTo', Url::current());
                    $authorizationURL = $this->ctctService->getAuthorizationUrl();
                    header("Location: $authorizationURL");
                    exit();

                }

            } catch(ConstantContactException $e) {

                $session->set('redirectTo', Url::current());
                $authorizationURL = $this->ctctService->getAuthorizationUrl();
                header("Location: $authorizationURL");
                exit();

            } catch(Exception $e) {

                // Log error
                Yii::error($e);

                // Show error message
                Yii::$app->getSession()->setFlash(
                    'danger',
                    $e->getMessage()
                );

            }

            /** @var User $currentUser */
            $currentUser = Yii::$app->user;
            $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
            $forms = ArrayHelper::map($forms, 'id', 'name');

            return $this->render('update', [
                'model' => $model,
                'fieldModel' => new ConstantContactField,
                'forms' => $forms,
                'lists' => $lists,
                'fields' => $fields,
            ]);
        }
    }

    /**
     * Enable / Disable multiple ConstantContact models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {

        $models = ConstantContact::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);

    }

    /**
     * Deletes an existing ConstantContact model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Finds the ConstantContact model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return ConstantContact the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    public function findModel($id)
    {
        if (($model = ConstantContact::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}