<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */
namespace app\modules\addons\modules\constant_contact;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormData;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\constant_contact\models\ConstantContact;
use app\modules\addons\modules\constant_contact\models\ConstantContactField;
use app\modules\addons\modules\constant_contact\services\ConstantContactException;
use app\modules\addons\modules\constant_contact\services\ConstantContactService;
use Exception;
use Yii;
use yii\helpers\Json;
use yii\helpers\Url;
use yii\web\Response;

/**
 * Class Module
 * @package app\modules\addons\modules\constant_contact
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "constant_contact";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';
    public $apiKey = 'f8fedbc9-d860-4f20-8755-238a0f03e7b4';
    public $consumerSecret = 'axy1YHR6TfMuoloceJ4VYg';
    public $redirectUrl;

    /**
     * @inheritdoc
     */
    public function init()
    {
        parent::init();

        $this->redirectUrl = Url::to(['/addons/constant_contact/admin/check'], true);
    }

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'ConstantContact' => ConstantContact::class,
        ];
    }

	/**
	 * Main Model of this Module
	 *
	 * @return \yii\db\ActiveQuery
	 */
	public function getActiveQuery()
	{
		return ConstantContact::find();
	}

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            'app.form.submission.verified' => function ($event) {
                $this->onSubmissionVerified($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            Form::class => [
                'beforeDelete' => [
                    [Module::class, 'onFormDeleted']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = ConstantContact::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new ConstantContact();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new ConstantContactField();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->constant_contact_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = ConstantContact::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Event Handler
     * When a form submission has been verified
     *
     * @param $event
     */
    public function onSubmissionVerified($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_VERIFIED);
    }

    /**
     * Make Request to API
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public function makeRequest($formModel, $submissionModel, $filePaths, $event)
    {

        $result = false;

        /** @var ConstantContact[] $models */
        $models = ConstantContact::findAll(['form_id' => $formModel->id, 'status' => 1]);
        /** @var FormData $dataModel */
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Form fields
        $fieldsForEmail = $dataModel->getFieldsForEmail();
        // Submission data in an associative array
        $fieldValues = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
        // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {

                /**
                 * Make API Request
                 */
                try {

                    $subscriber_email = isset($model->subscriber_email, $submissionData[$model->subscriber_email]) ?
                        $submissionData[$model->subscriber_email] : '';

                    if (!empty($subscriber_email)) {
                        /**
                         * Add Subscriber to Constant Contact List
                         */
                        $ctctService = new ConstantContactService($this->apiKey, $this->consumerSecret, $this->redirectUrl);

                        try {
                            // Refresh access token
                            $token = Json::decode($model->access_token, true);
                            if (isset($token['access_token'], $token['expires_in'], $token['unix_timestamp'])) {
                                if (time() > ((int) $token['unix_timestamp'] + (int) $token['expires_in'])) {
                                    $accessToken = $ctctService->refreshToken($token['refresh_token']);
                                    $token = Json::decode($accessToken, true);
                                }
                            }

                            if (isset($token['error'], $token['invalid_grant'])) {

                                // Log error
                                Yii::error([
                                    'addon' => 'Constant Contact',
                                    'error' => $token,
                                ]);

                            } else if (isset($token['access_token'])) {

                                // Save refresh token
                                if (isset($token['refresh_token'], $token['token_type'])) {
                                    $token['unix_timestamp'] = time();
                                    $model->access_token = Json::encode($token);
                                    $model->save();
                                }

                                // Use new access token
                                $ctctService->setToken($token['access_token']);
                                $attributes = [];
                                $customFields = [];
                                foreach ($model->items as $item) {
                                    // Get value
                                    $form_field = isset($item->form_field, $submissionData[$item->form_field]) ?
                                        $submissionData[$item->form_field] : '';
                                    // Check custom field value
                                    $fields = $ctctService->getFields();
                                    if (in_array($item->list_field, array_keys($fields))) {
                                        $attributes[$item->list_field] = $form_field;
                                    } else {
                                        $customFields[] = [
                                            'custom_field_id' => $item->list_field,
                                            'value' => $form_field,
                                        ];
                                    }
                                }

                                $contact = [
                                    'email_address' => $subscriber_email,
                                    'list_memberships' => [$model->list_id],
                                ];

                                if (!empty($attributes)) {
                                    $contact = array_merge($contact, $attributes);
                                }

                                if (!empty($customFields)) {
                                    $contact['custom_fields'] = $customFields;
                                }

                                // Add Subscriber
                                $result = $ctctService->syncContact($contact);
                            }

                        } catch(ConstantContactException $e) {

                            // Log error
                            Yii::error($e);

                            /** @var Response $response */
                            $response = Yii::$app->getResponse();
                            $response->format = Response::FORMAT_JSON;
                            $response->data = [
                                'action'  => 'submit',
                                'success' => false,
                                'id' => $formModel->id,
                                'message' => $e->getMessage(),
                            ];
                            $response->send();
                            exit;

                        } catch(Exception $e) {

                            // Log error
                            Yii::error($e);

                        }
                    }

                } catch (Exception $e) {

                    // Log exception
                    Yii::error($e);

                }
            }
        }

        return $result;

    }

}
