<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\conditional_email;

use app\components\rules\RuleEngine;
use app\components\User;
use app\controllers\AjaxController;
use app\events\SubmissionMailEvent;
use app\helpers\ArrayHelper;
use app\helpers\EventHelper;
use app\helpers\Liquid;
use app\helpers\MailHelper;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\conditional_email\models\ConditionalEmail;
use app\modules\addons\modules\conditional_email\models\ConditionalEmailTemplate;
use Yii;

/**
 * Class Module
 * @package app\modules\addons\modules\conditional_email
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "conditional_email";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'ConditionalEmail' => ConditionalEmail::class,
        ];
    }

    /**
     * Return ActiveQuery of ConditionalEmailTemplate models accessible by this user
     *
     * Important! Don't select columns with the returned active query,
     * because we can't get unexpected behaviors
     *
     * @return \yii\db\ActiveQuery
     */
    public function getCurrentUserEmailTemplates()
    {
        /** @var User $currentUser */
        $currentUser = Yii::$app->user;

        // If user can view all forms
        if ($currentUser->can("viewForms")) {
            // Return all conditional email templates
            return ConditionalEmailTemplate::find();
        }

        // Forms
        $formModels = $currentUser->forms()->asArray()->all();
        $formIds = ArrayHelper::getColumn($formModels, 'id');

        // Shared Conditional Email Templates
        $emailModels = ConditionalEmail::find()->where(['in', 'form_id', $formIds])->asArray()->all();
        $templateIds = ArrayHelper::getColumn($emailModels, 'template_id');

        // Own Conditional Email Templates
        $templates = ConditionalEmailTemplate::find()->where(['created_by' => $currentUser->id]);

        return $templates->union(ConditionalEmailTemplate::find()->where(['in', 'id', $templateIds]));
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            'app.form.submission.verified' => function ($event) {
                $this->onSubmissionVerified($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            'app\models\Form' => [
                'beforeDelete' => [
                    ['app\modules\addons\modules\conditional_email\Module', 'beforeDeleteAForm']
                ]
            ],
            'app\modules\addons\modules\pdf\models\Pdf' => [
                'init' => [
                    [self::class, 'onPdfInit']
                ]
            ],
            'app\modules\addons\modules\paypal\controllers\CheckController' => [
                EventHelper::EVENT_PAYPAL_PAYMENT_RECEIVED => [
                    [self::class, 'onPaypalPaymentReceived']
                ]
            ]
        ];
    }

    public static function onPdfInit($event)
    {
        $sender = $event->sender;
        if (!in_array(SubmissionMailEvent::EVENT_TYPE_CONDITIONAL, array_keys($sender->notificationTypes))) {
            $sender->notificationTypes[SubmissionMailEvent::EVENT_TYPE_CONDITIONAL] = Yii::t('app', 'Conditional Emails');
        }
        $event->sender = $sender;
    }
    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = ConditionalEmail::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new ConditionalEmail();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();
            }
        }
    }

    /**
     * Event Handler
     * Before a Form model is deleted
     *
     * @param $event
     * @throws \Throwable
     * @throws \yii\db\StaleObjectException
     */
    public static function beforeDeleteAForm($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $model = ConditionalEmail::findOne(['form_id' => $event->sender->id]);
            if ($model) {
                $model->delete();
            }
        }
    }

    /**
     * @param $event
     */
    public static function onPaypalPaymentReceived($event)
    {
        if (isset($event, $event->sender, $event->sender->payment)) {
            /** @var \app\modules\addons\modules\paypal\models\PaypalPayment $payment */
            $payment = $event->sender->payment;
            $submissionModel = $payment->submission;
            $formModel = empty($payment->form) ? $submissionModel->form : $payment->form;
            $filePaths = [];

            // If file paths are empty, find them by model relation
            if (empty($filePaths)) {
                $fileModels = $submissionModel->files;
                foreach ($fileModels as $fileModel) {
                    $filePaths[] = $fileModel->getUrl();
                }
            }

            /*******************************
            /* Send Conditional Emails
            /*******************************/
            self::sendConditionalEmails($formModel, $submissionModel, $filePaths, EventHelper::STATUS_PAYPAL_PAYMENT_RECEIVED);
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getUrl();
            }
        }

        /*******************************
        /* Send Conditional Emails
        /*******************************/
        self::sendConditionalEmails($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Event Handler
     * When a form submission has been verified
     *
     * @param $event
     */
    public function onSubmissionVerified($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getUrl();
            }
        }

        /*******************************
        /* Send Conditional Emails
        /*******************************/
        self::sendConditionalEmails($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_VERIFIED);
    }

    /**
     * Send conditional emails
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public static function sendConditionalEmails($formModel, $submissionModel, $filePaths, $event)
    {
        $result = false;

        $emails = ConditionalEmail::findAll(['form_id' => $formModel->id, 'status' => 1]);
        /** @var \app\models\FormData $dataModel */
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Form fields
        $fieldsForEmail = $dataModel->getFieldsForEmail();
        // Submission data in an associative array
        $tokens = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
        // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        // Async Email
        $async = MailHelper::async();

        // Sender by default: No-Reply Email
        $fromEmail = MailHelper::from(Yii::$app->settings->get("app.noreplyEmail"));

        // Sender verification
        if (empty($fromEmail)) {
            return false;
        }

        /*******************************
        /* Process Emails
        /*******************************/
        foreach ($emails as $email) {

            // Only when the required event occurs
            if ($email->event !== $event) {
                continue;
            }

            // By default, send email
            $isValid = true;

            // Conditional Logic
            if (!empty($email->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $email->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {
                // Email Views
                $emailViews = [
                    'html' => 'confirmation-html',
                    'text' => 'confirmation-text',
                ];

                // Text Only
                if ($email->mail_text_only) {
                    $emailViews['html'] = 'confirmation-text';
                }

                // Message
                $data = [
                    'fieldValues' => $tokens, // Submission data for replacement
                    'fieldData' => $fieldData, // Submission data for print details
                    'mail_receipt_copy' => (boolean) $email->mail_receipt_copy, // Add submission copy
                    'message' => SubmissionHelper::replaceSubmissionTableToken($email->template->body_html, $fieldData),
                ];

                // Subject
                $subject = isset($email->mail_subject) && !empty($email->mail_subject) ?
                    $email->mail_subject : Yii::t('app', 'Thanks for your message');

                // Token replacement in subject
                $subject = SubmissionHelper::replaceTokens($subject, $tokens);
                $subject = Liquid::render($subject, $tokens);

                // Add name to From
                if (!empty($email->mail_from_name)) {
                    $fromName = isset($submissionData[$email->mail_from_name]) ?
                        $submissionData[$email->mail_from_name] : $email->mail_from_name;
                    $fromName = is_array($fromName) ? implode(",", $fromName) : $fromName;

                    if (is_array($fromEmail)) {
                        $fromEmail = [
                            key($fromEmail) => $fromName,
                        ];
                    } else {
                        $fromEmail = [
                            $fromEmail => $fromName,
                        ];
                    }
                }

                // Compose email
                $message = Yii::$app->mailer->compose($emailViews, $data);

                // Disable Views
                if (strpos($email->template->body_html, "</html>") !== false) {
                    $htmlBody = SubmissionHelper::replaceTokens($email->template->body_html, $tokens);
                    $htmlBody = Liquid::render($htmlBody, $tokens);
                    $textBody = strip_tags($htmlBody);
                    Yii::$app->mailer->htmlLayout = false;
                    $message = Yii::$app->mailer
                        ->compose(null)
                        ->setHtmlBody($htmlBody)
                        ->setTextBody($textBody);
                }

                $message
                    ->setFrom($fromEmail)
                    ->setSubject($subject);

                // Add email address as email recipient
                if (!empty($email->mail_to)) {
                    // Validate e-mail
                    if (!filter_var($email->mail_to, FILTER_VALIDATE_EMAIL) === false) {
                        $message->setTo($email->mail_to);
                    }
                }

                // Add email field as email recipient
                if (!empty($email->field_to) && !empty($submissionData[$email->field_to])) {
                    $toEmailField = $submissionData[$email->field_to];
                    // Remove all illegal characters from email
                    $toEmailField = filter_var($toEmailField, FILTER_SANITIZE_EMAIL);
                    if (!filter_var($toEmailField, FILTER_VALIDATE_EMAIL) === false) {
                        if (!empty($message->getTo())) {
                            $message->setTo([$email->mail_to, $toEmailField]);
                        } else {
                            $message->setTo($toEmailField);
                        }
                    }
                }

                // If message has a recipient
                if (!empty($message->getTo())) {
                    // Reply to
                    if (!empty($email->mail_reply_to)) {
                        // ReplyTo (can be an email or an email field)
                        $replyToField = isset($submissionData[$email->mail_reply_to]) ? $submissionData[$email->mail_reply_to] : null;
                        $replyTo = $email->mailReplyToIsEmail() ? $email->mail_reply_to :
                            $replyToField;

                        // Add name to Reply to
                        if (!empty($replyTo) && !empty($email->mail_from_name)) {
                            $fromName = isset($submissionData[$email->mail_from_name]) ?
                                $submissionData[$email->mail_from_name] : $email->mail_from_name;
                            $fromName = is_array($fromName) ? implode(",", $fromName) : $fromName;

                            if (is_array($replyTo)) {
                                $replyTo = [
                                    key($replyTo) => $fromName,
                                ];
                            } else {
                                $replyTo = [
                                    $replyTo => $fromName,
                                ];
                            }
                        }

                        if (!empty($replyTo)) {
                            $message->setReplyTo($replyTo);
                        }
                    }

                    // Add cc
                    if (!empty($email->mail_cc)) {
                        $message->setCc(explode(',', $email->mail_cc));
                    }

                    // Add bcc
                    if (!empty($email->mail_bcc)) {
                        $message->setBcc(explode(',', $email->mail_bcc));
                    }

                    // Attach files
                    if ($email->mail_attachments && count($filePaths) > 0) {
                        foreach ($filePaths as $filePath) {
                            $message->attach(Yii::getAlias('@app') . DIRECTORY_SEPARATOR . $filePath);
                        }
                    }

                    // Trigger Event
                    Yii::$app->trigger(SubmissionMailEvent::EVENT_NAME, new SubmissionMailEvent([
                        'submission' => $submissionModel,
                        'type' => SubmissionMailEvent::EVENT_TYPE_CONDITIONAL,
                        'message' => $message,
                        'async' => $async,
                        'tokens' => $tokens,
                    ]));

                    // Send E-mail
                    if ($async) {
                        $result = $message->queue();
                    } else {
                        $result = $message->send();
                    }
                }
            }
        }

        return $result;
    }
}