<?php

namespace app\modules\addons\modules\cleverreach\services;

use Yii;
use Exception;
use yii\helpers\Json;

class CleverReachService
{

    /** @string Base Url */
    public $baseUrl = "https://rest.cleverreach.com/v3";

    /** @string Token Url */
    public $tokenUrl = "https://rest.cleverreach.com/oauth/token.php";

    /** @var CleverReachRestApi cr */
    protected $api;

    /** @string Client ID */
    protected $clientId;

    /** @string Client Secret */
    protected $clientSecret;

    /** @string Access Token */
    protected $accessToken;

    /** @string Access Token */
    protected $expiresIn;

    /** @string Access Token */
    protected $tokenType;

    /** @string Access Token */
    protected $scope;

    /** @string Authorization */
    protected $authorization;

    /**
     * CleverReachService constructor.
     * @param $clientID
     * @param $clientSecret
     */
    public function __construct($clientID, $clientSecret)
    {
        $this->clientId = $clientID;
        $this->clientSecret = $clientSecret;
        $this->api = new CleverReachRestApi($this->baseUrl);
    }

    /**
     * Refresh OAuth
     *
     * @return bool
     */
    protected function refreshAuthorization()
    {
        $curl = curl_init();
        curl_setopt($curl,CURLOPT_URL, $this->tokenUrl);
        curl_setopt($curl,CURLOPT_USERPWD, $this->clientId . ":" . $this->clientSecret);
        curl_setopt($curl,CURLOPT_POSTFIELDS, array("grant_type" => "client_credentials"));
        curl_setopt($curl,CURLOPT_RETURNTRANSFER, true);
        $result = curl_exec($curl);
        curl_close ($curl);

        if (!empty($result)) {
            $auth = Json::decode($result);
            if (isset($auth['access_token'], $auth['expires_in'], $auth['token_type'], $auth['scope'])) {
                $this->accessToken = $auth['access_token'];
                $this->expiresIn = $auth['expires_in'];
                $this->tokenType = $auth['token_type'];
                $this->scope = $auth['scope'];
                $auth['unix_timestamp'] = time();
                $this->authorization = Json::encode($auth);
                return true;
            }
        }

        return false;
    }

    /**
     * @return string Authorization
     */
    public function getAuthorization()
    {
        $this->refreshAuthorization();
        return $this->authorization;
    }

    /**
     * @param $accessToken
     */
    public function setAccessToken($accessToken)
    {
        $this->accessToken = $accessToken;
    }

    /**
     * @return mixed
     */
    public function getAccessToken()
    {
        return $this->accessToken;
    }

    /**
     * Add or Edit Subscriber
     *
     * @param $contact_list
     * @param $contact
     * @return array|false
     * @throws Exception
     */
    public function syncContact($contact_list, $contact)
    {
        $this->api->setAuthMode("bearer", $this->accessToken);
        return $this->api->post("/groups/{$contact_list}/receivers", $contact);
    }

    /**
     * Get Subscriber Lists
     *
     * @return array
     * @throws Exception
     */
    public function getSubscriberLists()
    {
        $this->api->setAuthMode("bearer", $this->accessToken);
        $listCollection  =  (array) $this->api->get("/groups");

        $lists = [];
        if (is_array($listCollection) && !empty($listCollection)) {
            foreach($listCollection as $list) {
                $list = (array) $list;
                if (isset($list['id'], $list['name'])) {
                    $lists[$list['id']] = $list['name'];
                }
            }
        }

        return $lists;
    }

    /**
     * Get Custom Fields
     *
     * @return array
     * @throws Exception
     */
    public function getCustomFields()
    {
        $this->api->setAuthMode("bearer", $this->accessToken);
        $customFieldCollection  =  (array) $this->api->get("/attributes");

        $fields = [];
        if (is_array($customFieldCollection) && !empty($customFieldCollection)) {
            foreach($customFieldCollection as $customField) {
                $customField = (array) $customField;
                if (isset($customField['name'], $customField['description'])) {
                    $fields[$customField['name']] = $customField['description'];
                }
            }
        }

        return $fields;
    }

    /**
     * Get Forms
     *
     * @param $id
     * @return array
     * @throws Exception
     */
    public function getForms($id)
    {
        $this->api->setAuthMode("bearer", $this->accessToken);
        $formCollection  =  (array) $this->api->get("/groups/{$id}/forms");

        $forms = [];
        if (is_array($formCollection) && !empty($formCollection)) {
            foreach($formCollection as $form) {
                $form = (array) $form;
                if (isset($form['id'], $form['name'])) {
                    array_push($forms, [
                        'id' => $form['id'],
                        'name' => $form['name'],
                    ]);
                }
            }
        }

        return $forms;
    }

    /**
     * @param $form_id
     * @param $email
     * @return mixed
     * @throws Exception
     */
    public function sendDOIEmail($form_id, $email)
    {
        $this->api->setAuthMode("bearer", $this->accessToken);
        return $this->api->post("/forms/{$form_id}/send/activate", array(
            "email"   => $email,
            "doidata" => array(
                "user_ip"    => $_SERVER["REMOTE_ADDR"],
                "referer"    => $_SERVER["SERVER_NAME"].$_SERVER["REQUEST_URI"],
                "user_agent" => $_SERVER["HTTP_USER_AGENT"]
            )
        ));
    }

}