<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\braintree;

use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\Html;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormData;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\braintree\models\Braintree;
use app\modules\addons\modules\braintree\models\BraintreeItem;
use app\modules\addons\modules\braintree\models\BraintreePayment;
use app\modules\addons\modules\braintree\services\BraintreeService;
use Exception;
use Yii;
use yii\helpers\Json;
use yii\web\Response;

/**
 * Class Module
 * @package app\modules\addons\modules\braintree
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "braintree";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'Braintree' => Braintree::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            'yii\base\View' => [
                'afterRender' => [
                    ['app\modules\addons\modules\braintree\Module', 'onViewAfterRender']
                ],
            ],
            'app\models\Form' => [
                'beforeDelete' => [
                    ['app\modules\addons\modules\braintree\Module', 'onFormDeleted']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = Braintree::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new Braintree();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new BraintreeItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->braintree_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a form model is deleted
     *
     * @param $event
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = Braintree::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * After a view is rendered
     *
     * @param $event
     */
    public static function onViewAfterRender($event)
    {
        if (isset($event, $event->sender, $event->sender->context) &&
            isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
            $event->sender->context->module->requestedRoute === "app/embed"
        ) {

            $formModel = $event->sender->context->getFormModel();
            $model = Braintree::findOne(['form_id' => $formModel->id, 'status' => 1]);

            if ($model && !empty($model->public_key)) {

                // Client settings
                $service = new BraintreeService(
                    $model->environment,
                    $model->merchant_id,
                    $model->public_key,
                    $model->private_key
                );

                $authorization = $service->getAuthorization();
                $required = $model->required ? 'true' : 'false';
                $requiredCssClass = $model->required ? 'required-control' : '';

                // Label
                $label = empty($model->label) ? Yii::t('app', 'Credit or debit card') : $model->label;
                $label = Html::label($label, 'braintree', ['class' => 'control-label braintree-card-label']);
                if ($model->hide_label) {
                    $label = '';
                }

                // HTML Code
                $htmlCode = <<<HTML
<div id="braintree-container" class="form-group {$requiredCssClass}">
    <style type="text/css">
        .braintree-placeholder {
            height: 0;
            margin: 0;
        }
    </style>
    {$label}  
    <div id="braintree"></div>
</div>
HTML;

                // Card Field settings
                $settings = [
                    'authorization' => $authorization,
                    'container' => '#braintree',
                    'locale' => $model->locale,
                    'vaultManager' => true,
                ];

                if ($model->cardholder_name) {
                    // Cardholder name
                    $settings['card'] = [
                        'cardholderName' => [
                            'required' => (boolean) $model->cardholder_name_required,
                        ]
                    ];
                }
                $settings = json_encode($settings);

                // JS Code
                $jsCode = <<<EOT
<script src="//js.braintreegateway.com/web/dropin/1.18.0/js/dropin.min.js"></script>
<script>
$(document).ready(function () {
    var dropinInstance;
    var submitButton = $('button[type=submit]');
    var paymentMethodRequestable = false;
    var settings = JSON.parse('{$settings}');
    var braintreeNonceHandler = function (nonce) {
        var hiddenInput = document.createElement('input');
        hiddenInput.setAttribute('type', 'hidden');
        hiddenInput.setAttribute('name', 'braintree-nonce');
        hiddenInput.setAttribute('value', nonce);
        formEl.find('[name="braintree-nonce"]').remove();
        formEl.append(hiddenInput);
        formEl.submit();
    }
    braintree.dropin.create(settings, function (err, instance) {
        if (err) {
            return;
        }
        dropinInstance = instance;
        dropinInstance.on('paymentMethodRequestable', function () {
            paymentMethodRequestable = true;
        });
        dropinInstance.on('noPaymentMethodRequestable', function () {
            paymentMethodRequestable = false;
        });
    });
    submitButton.click(function(e) {
        if (paymentMethodRequestable || {$required}) {
            e.preventDefault();
            dropinInstance.requestPaymentMethod(function (err, payload) {
                if (err) {
                    dropinInstance.clearSelectedPaymentMethod();
                    return;
                }
                braintreeNonceHandler(payload.nonce);
            });
        }
    });
    formEl.on('success', function(){ 
        dropinInstance.teardown(function(err) {
            if (err) { console.error('An error occurred during teardown:', err); }
        });
        $('#braintree-container').hide();
     }); 
})
</script>
</body>
EOT;
                $content = $event->output;
                $content = str_replace("{{BRAINTREE}}", $htmlCode, $content);
                $event->output = str_replace("</body>", $jsCode, $content);
            }
        }
    }

    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Make Request to API
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public function makeRequest($formModel, $submissionModel, $filePaths, $event)
    {

        $result = false;

        $models = Braintree::findAll(['form_id' => $formModel->id, 'status' => 1]);
        /** @var FormData $dataModel */
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Form fields
        $fieldsForEmail = $dataModel->getFieldsForEmail();
        // Submission data in an associative array
        $fieldValues = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
        // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {

                /**
                 * Make API Request
                 */
                try {

                    if ($nonce = Yii::$app->request->post('braintree-nonce')) {

                        // Amount
                        $amount = 0;
                        $customFields = [];

                        foreach ($model->items as $itemModel) {
                            // Check field value
                            $quantity = isset($itemModel->quantity, $data[$itemModel->quantity]) &&
                            !empty($data[$itemModel->quantity]) ?
                                $data[$itemModel->quantity] : 1;
                            $price = isset($itemModel->price, $data[$itemModel->price]) &&
                            !empty($data[$itemModel->price]) ?
                                $data[$itemModel->price] : 0;

                            // Sum values if field value is an array
                            $quantity = is_array($quantity) ? array_sum($quantity) : $quantity;
                            $price = is_array($price) ? array_sum($price) : $price;

                            $amount += $quantity * $price;
                            if (!empty($itemModel->api_name)) {
                                $customFields[$itemModel->api_name] = Yii::t('app', 'Description:' ) . $itemModel->description . ' - '. Yii::t('app', 'Price:').' ' . $price . ' - Quantity: ' . $quantity . ' - SubTotal: ' . number_format( $quantity * $price, 2);
                            }
                        }

                        $service = new BraintreeService(
                            $model->environment,
                            $model->merchant_id,
                            $model->public_key,
                            $model->private_key
                        );

                        // Sale
                        $sale = [
                            'amount' => $amount,
                            'paymentMethodNonce' => $nonce,
                            'orderId' => sprintf("%s - Form: #%s - Submission: #%s", Yii::$app->settings->get('app.name'), $model->form_id, $submissionModel->id),
                            'customer' => [
                                'firstName' => !empty($data[$model->first_name]) ? $data[$model->first_name] : '',
                                'lastName' => !empty($data[$model->last_name]) ? $data[$model->last_name] : '',
                                'company' => !empty($data[$model->company]) ? $data[$model->company] : '',
                                'phone' => !empty($data[$model->phone]) ? $data[$model->phone] : '',
                                'website' => !empty($data[$model->website]) ? $data[$model->website] : '',
                                'email' => !empty($data[$model->email]) ? $data[$model->email] : '',
                            ],
                            'billing' => [
                                'streetAddress' => !empty($data[$model->address_line1]) ? $data[$model->address_line1] : '',
                                'extendedAddress' => !empty($data[$model->address_line2]) ? $data[$model->address_line2] : '',
                                'locality' => !empty($data[$model->address_city]) ? $data[$model->address_city] : '',
                                'region' => !empty($data[$model->address_state]) ? $data[$model->address_state] : '',
                                'postalCode' => !empty($data[$model->address_zip]) ? $data[$model->address_zip] : '',
                                'countryCodeAlpha2' => !empty($data[$model->address_country]) ? $data[$model->address_country] : '',
                            ],
                            'customFields' => $customFields,
                            'options' => [
                                'submitForSettlement' => true,
                                'storeInVaultOnSuccess' => true,
                                'addBillingAddressToPaymentMethod' => true,
                            ]
                        ];

                        // Submit the sale
                        $result = $service->sale($sale);

                        if ($result->success || !empty($result->transaction)) {
                            $transaction = $result->transaction;
                            if ($transaction && isset($transaction->id)) {
                                $payment = new BraintreePayment();
                                $payment->braintree_id = $model->id;
                                $payment->form_id = $formModel->id;
                                $payment->submission_id = $submissionModel->id;
                                $payment->transaction = $transaction->id;
                                $payment->data = Json::encode($transaction);
                                $payment->save(false);
                            }
                        } else {
                            $errorString = "";
                            foreach($result->errors->deepAll() as $error) {
                                $errorString .= "<p>" . $error->message . "</p>";
                            }
                            /** @var \yii\web\Response $response */
                            $response = Yii::$app->getResponse();
                            $response->format = Response::FORMAT_JSON;
                            $response->data = array(
                                'action'  => 'submit',
                                'success' => false,
                                'id' => $formModel->id,
                                'message' => !empty($errorString) ? $errorString : Yii::t('app', 'An unexpected error has occurred. Please retry later.'),
                            );
                            $response->send();
                            exit;
                        }
                    }

                } catch (Exception $e) {

                    // Log exception
                    Yii::error($e);

                }
            }
        }

        return $result;

    }

}