<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.1
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\aweber\services;

use app\modules\addons\modules\aweber\Module;
use Yii;

/**
 * Class AweberService
 * @package app\modules\addons\modules\aweber\services
 */
class AweberService
{
    protected $baseUrl = "https://api.aweber.com/1.0/";
    protected $authorizedUrl = "https://auth.aweber.com/oauth2/authorize";
    protected $tokenUrl = "https://auth.aweber.com/oauth2/token";
    protected $redirectUrl = "urn:ietf:wg:oauth:2.0:oob";
    protected $clientId;
    protected $userSession;
    protected $client;
    protected $accessToken;

    protected $scopes = array(
        "account.read",
        "list.read",
        "list.write",
        "subscriber.read",
        "subscriber.write",
        "email.read",
        "email.write",
        "subscriber.read-extended"
    );

    public function __construct()
    {
        require_once(Yii::getAlias('@addons/aweber/services/vendor/autoload.php'));

        $aweberModule = Module::getInstance();
        $this->clientId = $aweberModule->clientID;
        $this->userSession = Yii::$app->session;
        $this->client = new \GuzzleHttp\Client();
    }

    /**
     * Get Oauth 2.0 Authorization Url
     *
     * @return string
     * @throws \Exception
     */
    public function getAuthorizationUrl()
    {
        // Generate the code challenge using the OS / cryptographic random function
        $verifierBytes = random_bytes(64);
        $codeVerifier = rtrim(strtr(base64_encode($verifierBytes), "+/", "-_"), "=");
        $this->userSession->set('aweber_code_verifier', $codeVerifier);

        // Very important, "raw_output" must be set to true or the challenge
        // will not match the verifier.
        $challengeBytes = hash("sha256", $codeVerifier, true);
        $codeChallenge = rtrim(strtr(base64_encode($challengeBytes), "+/", "-_"), "=");

        // State token, a uuid is fine here
        $state = uniqid();

        // Assemble the authorize URL and direct the user to a browser
        // to sign in to their AWeber customer account
        $authorizeQuery = array(
            "response_type" => "code",
            "client_id" => $this->clientId,
            "redirect_uri" => $this->redirectUrl,
            "scope" => implode(" ",$this->scopes),
            "state" => $state,
            "code_challenge" => $codeChallenge,
            "code_challenge_method" => "S256"
        );

        return $this->authorizedUrl . "?" . http_build_query($authorizeQuery);
    }

    /**
     * Get Oauth 2.0 credentials
     *
     * @param $authorizationCode
     * @return string
     */
    public function getOauthCredentials($authorizationCode)
    {
        $credentials = [];
        if ($this->userSession->has('aweber_code_verifier')) {
            $codeVerifier = $this->userSession->get('aweber_code_verifier');
            // Use the authorization code to fetch an access token
            $tokenQuery = array(
                "grant_type" => "authorization_code",
                "code" => $authorizationCode,
                "client_id" => $this->clientId,
                "code_verifier" => $codeVerifier,
            );

            // Create a new GuzzleHTTP Client and define scopes
            // If you need different scopes, modify them here.
            $tokenUrl = $this->tokenUrl . "?" . http_build_query($tokenQuery);
            $response = $this->client->post($tokenUrl);

            // Get the credentials
            $credentials = json_decode($response->getBody(), true);
            $credentials['unix_timestamp'] = time();

            $this->userSession->remove('aweber_code_verifier');
        }

        return json_encode($credentials, true);
    }

    /**
     * Refresh Oauth 2.0 credentials
     *
     * @param $oauth
     * @return false|string
     */
    public function refreshOauthCredentials($oauth)
    {
        if (is_string($oauth)) {
            $oauth = json_decode($oauth, true);
        }

        $client = new \GuzzleHttp\Client();
        $refreshToken = $oauth['refresh_token'];
        $response = $client->post(
            $this->tokenUrl, [
                'form_params' => [
                    'client_id' => $this->clientId,
                    'grant_type' => 'refresh_token',
                    'refresh_token' => $refreshToken,
                ]
            ]
        );
        $credentials = json_decode($response->getBody(), true);
        $credentials['unix_timestamp'] = time();

        return json_encode($credentials, true);
    }

    /**
     * Set Access Token
     *
     * @param $accessToken
     */
    public function setAccessToken($accessToken)
    {
        $this->accessToken = $accessToken;
    }

    /**
     * Get all of the entries for a collection
     *
     * @param string $url Full url to make the request
     * @return array Every entry in the collection
     */
    public function getCollection($url) {

        $collection = array();

        while (isset($url)) {
            $request = $this->client->get($url, [
                'headers' => [
                    'User-Agent' => 'Easy Forms',
                    'Accept' => 'application/json',
                    'Authorization' => 'Bearer ' . $this->accessToken,
                ]]);
            $body = $request->getBody();
            $page = json_decode($body, true);
            $collection = array_merge($page['entries'], $collection);
            $url = isset($page['next_collection_link']) ? $page['next_collection_link'] : null;
        }

        return $collection;
    }

    /**
     * Get Account Url
     *
     * @return mixed
     */
    public function getAccountUrl()
    {
        $accounts = $this->getCollection($this->baseUrl . 'accounts');
        return $accounts[0]['self_link'];
    }

    /**
     * Get Lists
     *
     * @return array
     */
    public function getSubscriberLists()
    {
        $accountUrl = $this->getAccountUrl();
        $findUrl = $accountUrl . '/lists';
        $collection = $this->getCollection($findUrl);

        $items = [];
        foreach($collection as $item) {
            $items[$item['id']] = $item['name'];
        }
        return $items;
    }

    /**
     * Get Custom Fields from List
     *
     * @param $listID
     * @return array
     */
    public function getCustomFields($listID)
    {
        $accountUrl = $this->getAccountUrl();
        $findUrl = $accountUrl . "/lists/{$listID}/custom_fields";
        $collection = $this->getCollection($findUrl);

        $items = [];
        foreach($collection as $item) {
            array_push($items, [
                'id' => $item['name'],
                'name' => $item['name'],
            ]);
        }
        return $items;
   }

   public function findSubscriber($listID, $email)
   {
       // /accounts/{accountId}/lists/{listId}/subscribers?ws.op=find
       $accountUrl = $this->getAccountUrl();
       $findUrl = $accountUrl . "/lists/{$listID}/subscribers";
       $params = array(
           'ws.op' => 'find',
           'email' => $email  // email to search for
       );
       $findUrl = $findUrl . '?' . http_build_query($params);
       return $this->getCollection($findUrl);
   }

    public function addSubscriber($listID, $params)
    {
        $accountUrl = $this->getAccountUrl();
        $subsUrl = $accountUrl . "/lists/{$listID}/subscribers";
        $subscriber = null;
        if (isset($params['email'])) {
            $foundSubscribers = $this->findSubscriber($listID, $params['email']);
            if (isset($foundSubscribers[0]['self_link'])) {
                if (isset($params['tags'])) {
                    $tags = $params['tags'];
                    $params['tags'] = [
                        'add' => $tags,
                    ];
                }
                // update the subscriber if they are on the first list
                $subscriberUrl = $foundSubscribers[0]['self_link'];
                $subscriberResponse = $this->client->patch($subscriberUrl, [
                    'json' => $params,
                    'headers' => ['Authorization' => 'Bearer ' . $this->accessToken]
                ])->getBody();
                $subscriber = json_decode($subscriberResponse, true);
            } else {
                // add the subscriber if they are not already on the first list
                $body = $this->client->post($subsUrl, [
                    'json' => $params,
                    'headers' => ['Authorization' => 'Bearer ' . $this->accessToken]
                ]);

                // get the subscriber entry using the Location header from the post request
                $subscriberUrl = $body->getHeader('Location')[0];
                $subscriberResponse = $this->client->get($subscriberUrl,
                    ['headers' => ['Authorization' => 'Bearer ' . $this->accessToken]])->getBody();
                $subscriber = json_decode($subscriberResponse, true);
            }
        }
        return $subscriber;
    }
}