<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\modules\addons\modules\aweber\controllers;

use app\components\User;
use app\modules\addons\modules\aweber\Module;
use kartik\depdrop\DepDropAction;
use Yii;
use yii\helpers\Json;
use yii\helpers\Url;
use app\models\Form;
use app\models\FormData;
use yii\web\Controller;
use yii\filters\VerbFilter;
use app\helpers\ArrayHelper;
use yii\web\NotFoundHttpException;
use app\modules\addons\modules\aweber\models\Aweber;
use app\modules\addons\modules\aweber\models\AweberSearch;
use app\modules\addons\modules\aweber\models\AweberField;
use app\modules\addons\modules\aweber\services\AweberService;
use Exception;
use yii\filters\AccessControl;

/**
 * AdminController implements the CRUD actions for Aweber model.
 */
class AdminController extends Controller
{

    /** @var AweberService $aweberService */
    public $aweberService;

    public function init()
    {
        $this->aweberService = new AweberService();

        parent::init();
    }

    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['post'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'aweber-fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create', 'authorize'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('Aweber')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = Aweber::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @return array
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\aweber\models\Aweber',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'aweber-fields' => [
                'class' => DepDropAction::class,
                'outputCallback' => function ($id, $params) {

                    $output = array();
                    $formID = isset($params[0]) ? $params[0] : 0;
                    $credentials = null;
                    $model = Aweber::findOne(['form_id' => $formID, 'list_id' => $id]);
                    if (!empty($model->form)) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {
                            $credentials = json_decode($model->oauth, true);
                        }
                    } else {
                        $session = Yii::$app->session;
                        if ($session->has('aweber_oauth')) {
                            $credentials = json_decode($session->get('aweber_oauth'), true);
                        }
                    }
                    if (!empty($credentials['access_token'])) {
                        $this->aweberService->setAccessToken($credentials['access_token']);
                        $output = $this->aweberService->getCustomFields($id);
                    }
                    return $output;
                },
                'selectedCallback' => function ($listID, $params) {
                    if (isset($params[1]) && !empty($params[1])) {
                        return $params[1];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all Aweber models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new AweberSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single Aweber model.
     * @param integer $id
     * @return mixed
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new Aweber model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {

        $lists = [];
        $connected = false;

        $request = Yii::$app->request;
        $model = new Aweber();

        if ($request->isPost && $model->load($request->post())) {
            try {
                $session = Yii::$app->session;

                // Check oauth
                if (empty($model->oauth) && !empty($model->authorization_code)) {
                    $credentials = $this->aweberService->getOauthCredentials($model->authorization_code);
                    $model->oauth = json_encode(json_decode($credentials, true), true);
                }

                // Get AWeber lists
                if (!empty($model->oauth)) {
                    $credentials = json_decode($model->oauth, true);
                    $this->aweberService->setAccessToken($credentials['access_token']);
                    $lists = $this->aweberService->getSubscriberLists();
                }

                // Save credentials in User session
                $session->set('aweber_oauth', $model->oauth);
                $connected = true;

                /**
                 * Save Aweber model and related items
                 */
                if ($model->subscriber_email && $model->validate()) {
                    $model->items = Yii::$app->request->post('AweberField',[]);
                    $model->save(false);

                    // Remove credentials in User session
                    $session->remove('aweber_oauth');

                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The sign up form has been successfully configured.')
                    );
                    return $this->redirect(['index']);
                }

            } catch (Exception $e) {
                // Log
                Yii::error($e);
                // Show error message
                Yii::$app->getSession()->setFlash('danger', $e->getMessage());
            }
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'fieldModel' => new AweberField,
            'forms' => $forms,
            'lists' => $lists,
            'connected' => $connected,
        ]);
    }

    /**
     * Updates an existing Aweber model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param integer $id
     * @return mixed
     */
    public function actionUpdate($id)
    {
        $lists = [];
        $connected = false;

        $request = Yii::$app->request;
        $model = $this->findModel($id);

        try {

            if ($model->load($request->post()) && $model->validate()) {
                $items = Yii::$app->request->post('AweberField',[]);
                if (empty($items) && !empty($model->authorization_code)) {
                    $credentials = $this->aweberService->getOauthCredentials($model->authorization_code);
                    $model->oauth = json_encode(json_decode($credentials, true), true);
                } else {
                    $model->items = Yii::$app->request->post('AweberField',[]);
                }
                $model->save();
                Yii::$app->getSession()->setFlash(
                    'success',
                    Yii::t('app', 'The sign up form has been successfully updated.')
                );
                return $this->redirect(['index']);
            }

            // Get AWeber lists
            if (!empty($model->oauth)) {

                // Refresh access token if it has expired
                $oauth = Json::decode($model->oauth);
                if (isset($oauth['access_token'], $oauth['expires_in'], $oauth['unix_timestamp'])) {
                    if (time() > ((int) $oauth['unix_timestamp'] + (int) $oauth['expires_in'])) {
                        $model->oauth = $this->aweberService->refreshOauthCredentials($oauth);
                        $model->save();
                    }
                }

                $credentials = json_decode($model->oauth, true);
                $this->aweberService->setAccessToken($credentials['access_token']);
                $lists = $this->aweberService->getSubscriberLists();
            }

            // Save credentials in User session
            $session = Yii::$app->session;
            $session->set('aweber_oauth', $model->oauth);
            $connected = true;

        } catch (Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'fieldModel' => new AweberField,
            'forms' => $forms,
            'lists' => $lists,
            'connected' => $connected,
        ]);
    }

    /**
     * Redirect to AWeber's Authorization Page
     */
    public function actionAuthorize()
    {
        /** @var Module $module */
        $module = Yii::$app->controller->module;
        $authorizationURL = $this->aweberService->getAuthorizationUrl($module->clientID);
        header("Location: $authorizationURL");
        exit();
    }

    /**
     * Enable / Disable multiple Aweber models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {

        $models = Aweber::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        }

        foreach ($models as $model) {
            $model->status = $status;
            $model->update();
        }

        Yii::$app->getSession()->setFlash(
            'success',
            Yii::t('app', 'The selected items have been successfully updated.')
        );

        return $this->redirect(['index']);
    }

    /**
     * Deletes an existing Aweber model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Finds the Aweber model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return Aweber the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = Aweber::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}