<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 */
namespace app\modules\addons\modules\adyen;

use Adyen\Client;
use Adyen\Environment;
use Adyen\Service\Checkout;
use app\components\rules\RuleEngine;
use app\controllers\AjaxController;
use app\helpers\Html;
use app\helpers\SubmissionHelper;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\EventManagerInterface;
use app\modules\addons\FormManagerInterface;
use app\modules\addons\modules\adyen\models\Adyen;
use app\modules\addons\modules\adyen\models\AdyenItem;
use app\modules\addons\modules\adyen\models\AdyenPayment;
use Exception;
use Yii;
use yii\helpers\Json;

/**
 * Class Module
 * @package app\modules\addons\modules\adyen
 */
class Module extends \yii\base\Module implements EventManagerInterface, FormManagerInterface
{

    public $id = "adyen";
    public $defaultRoute = 'admin/index';
    public $controllerLayout = '@app/views/layouts/main';

    /**
     * @inheritdoc
     */
    public function getDefaultModelClasses()
    {
        return [
            'Adyen' => Adyen::class,
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachGlobalEvents()
    {
        return [
            'app.form.submission.accepted' => function ($event) {
                $this->onSubmissionAccepted($event);
            },
            AjaxController::EVENT_FORM_COPIED => function ($event) {
                $this->onFormCopied($event);
            },
        ];
    }

    /**
     * @inheritdoc
     */
    public function attachClassEvents()
    {
        return [
            'yii\base\View' => [
                'afterRender' => [
                    ['app\modules\addons\modules\adyen\Module', 'onViewAfterRender']
                ],
            ],
            'app\models\Form' => [
                'beforeDelete' => [
                    ['app\modules\addons\modules\adyen\Module', 'onFormDeleted']
                ]
            ],
        ];
    }

    /**
     * Event Handler
     * When a Form is Copied
     *
     * @param $event
     */
    public function onFormCopied($event)
    {
        if (isset($event, $event->form, $event->form->id, $event->oldForm, $event->oldForm->id)) {
            $oModels = Adyen::findAll(['form_id' => $event->oldForm->id]);
            foreach ($oModels as $oModel) {
                $model = new Adyen();
                $model->attributes = $oModel->attributes;
                $model->id = null;
                $model->form_id = $event->form->id;
                $model->isNewRecord = true;
                $model->save();

                foreach ($oModel->items as $oItem) {
                    $item = new AdyenItem();
                    $item->attributes = $oItem->attributes;
                    $item->id = null;
                    $item->adyen_id = $model->id;
                    $item->form_id = $event->form->id;
                    $item->isNewRecord = true;
                    $item->save();
                }
            }
        }
    }

    /**
     * Event Handler
     * Before a Form model is deleted
     * @param $event
     *
     * @throws \Exception
     * @throws \yii\db\StaleObjectException
     */
    public static function onFormDeleted($event)
    {
        if (isset($event) && isset($event->sender) && $event->sender instanceof Form && isset($event->sender->id)) {
            $models = Adyen::find()->where(['form_id' => $event->sender->id])->all();
            foreach ($models as $model) {
                $model->delete();
            }
        }
    }

    /**
     * Event Handler
     * After a view is rendered
     *
     * @param $event
     */
    public static function onViewAfterRender($event)
    {
        if (isset($event, $event->sender, $event->sender->context) &&
            isset($event->sender->context->module, $event->sender->context->module->requestedRoute) &&
            $event->sender->context->module->requestedRoute === "app/embed"
        ) {

            $formModel = $event->sender->context->getFormModel();
            $model = Adyen::findOne(['form_id' => $formModel->id, 'status' => 1]);

            if ($model && !empty($model->origin_key) && !empty($model->api_key) && !empty($model->dom_element)) {

                $loadingContext = rtrim($model->loading_context,"/");

                $headCode = <<<EOT

    <script src="{$loadingContext}/sdk/2.5.0/adyen.js"></script>
    <link rel="stylesheet" href="{$loadingContext}/sdk/2.5.0/adyen.css" />
</head>
EOT;

                $content = $event->output;
                $event->output = str_replace("</head>", $headCode, $content);

                $defaultErrorMessage = Yii::t('app', 'Invalid Credit Card');

                // Required
                $required = $model->required ? 'true' : 'false';
                $requiredCssClass = $model->required ? 'required-control' : '';

                // Label
                $label = empty($model->label) ? Yii::t('app', 'Credit card') : $model->label;
                $label = Html::label($label, $model->dom_element, ['class' => 'control-label adyen-card-label']);
                if ($model->hide_label) {
                    $label = '';
                }

                // Card Element
                $card = Html::tag('div', '', ['id' => $model->dom_element]);

                // HTML Code
                $htmlCode = <<<HTML
<div id="adyen-container" class="form-group {$requiredCssClass}">
    {$label}  
    {$card}
    <div id="adyen-card-errors" class="help-block error-block"></div>                
</div>
HTML;

                $locale = $model->locale;
                $originKey = $model->origin_key;
                $loadingContext = $model->loading_context;
                $domElement = "#" . $model->dom_element;
                $storeDetails = $model->store_details ? 'true' : 'false';
                $holderName = $model->holder_name ? 'true' : 'false';
                $styles = $model->styles;
                $placeholders = $model->placeholders;

                $jsCode = <<<EOT

<script>
$(document).ready(function(){

    var configuration = {
        locale: "{$locale}",
        originKey: "{$originKey}",
        loadingContext: "{$loadingContext}"
    };

    var checkout = new AdyenCheckout(configuration);

    var card = checkout.create("card", {
        enableStoreDetails: {$storeDetails},
        hasHolderName: {$holderName},
        styles: {$styles},
        placeholders: {$placeholders},
        onChange: adyenHandlerOnChange
    }).mount("{$domElement}");
    
    var isValidCard = false;

    function adyenHandlerOnChange(state) {
        // Remove field
        var hiddenInput = document.getElementById('adyen-state');
        if (hiddenInput) {
            hiddenInput.parentNode.removeChild(hiddenInput);            
        } 
        if (state.isValid) {
            // Add field
            hiddenInput = document.createElement('input');
            hiddenInput.setAttribute('type', 'hidden');
            hiddenInput.setAttribute('id', 'adyen-state');
            hiddenInput.setAttribute('name', 'adyen-state');
            hiddenInput.setAttribute('value', JSON.stringify(state.data));
            formEl.find('[name="adyen-state"]').remove();
            formEl.append(hiddenInput);
            isValidCard = true;
        } else {
            isValidCard = false;
        }
        Utils.postMessage({
            height: $("body").outerHeight(true)
        });
    }
    
    $('button[type=submit]').click(function(e) {
        if ({$required}) {
            e.preventDefault();
            if (isValidCard) {
                $('#adyen-container').removeClass('has-error');
                $('#adyen-card-errors').text('')
                formEl.submit();
            } else {
                $('#adyen-container').removeClass('has-error').addClass('has-error');
                $('#adyen-card-errors').text('{$defaultErrorMessage}')
            }
        }
    });
    
    formEl.on('success', function(event){ 
        $('{$domElement}').parent().hide();
     }); 
})
</script>
</body>
EOT;

                $content = $event->output;
                $content = str_replace("{{ADYEN}}", $htmlCode, $content);
                $event->output = str_replace("</body>", $jsCode, $content);

            }
        }
    }


    /**
     * Event Handler
     * When a form submission has been accepted
     *
     * @param $event
     */
    public function onSubmissionAccepted($event)
    {
        /** @var FormSubmission $submissionModel */
        $submissionModel = $event->submission;
        /** @var Form $formModel */
        $formModel = empty($event->form) ? $submissionModel->form : $event->form;
        /** @var array $filePaths */
        $filePaths = empty($event->filePaths) ? [] : $event->filePaths;

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getLink();
            }
        }

        /*******************************
        /* Make API Request
        /*******************************/
        $this->makeRequest($formModel, $submissionModel, $filePaths, FormSubmission::STATUS_ACCEPTED);
    }

    /**
     * Make Request to API
     *
     * @param $formModel
     * @param $submissionModel
     * @param array $filePaths
     * @param int $event Event Type
     * @return bool
     */
    public function makeRequest($formModel, $submissionModel, $filePaths, $event)
    {

        $result = false;

        $models = Adyen::findAll(['form_id' => $formModel->id, 'status' => 1]);
        /** @var \app\models\FormData $dataModel */
        $dataModel = $formModel->formData;
        /** @var array $submissionData */
        $submissionData = $submissionModel->getSubmissionData();
        // Form fields
        $fieldsForEmail = $dataModel->getFieldsForEmail();
        // Submission data in an associative array
        $fieldValues = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
        // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
        $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);
        // Submission data for rule engine
        $data = SubmissionHelper::prepareDataForRuleEngine($submissionModel->data, $dataModel->getFields());

        /*******************************
        /* Process
        /*******************************/
        foreach ($models as $model) {

            // Only when the required event occurs
            if ($model->event !== $event) {
                continue;
            }

            // By default
            $isValid = true;

            // Conditional Logic
            if (!empty($model->conditions)) {
                $engine = new RuleEngine([
                    'conditions' => $model->conditions,
                    'actions' => [],
                ]);
                $isValid = $engine->matches($data);
            }

            // If the conditions have been met
            if ($isValid) {

                /**
                 * Make API Request
                 */
                try {

                    if ($state = Yii::$app->request->post('adyen-state')) {

                        $state = Json::decode($state);

                        // Amount
                        $amount = 0;
                        foreach ($model->items as $itemModel) {
                            // Check field value
                            /** @var AdyenItem $itemModel */
                            $quantity = isset($itemModel->quantity, $data[$itemModel->quantity]) &&
                            !empty($data[$itemModel->quantity]) ?
                                $data[$itemModel->quantity] : 1;
                            $price = isset($itemModel->price, $data[$itemModel->price]) &&
                            !empty($data[$itemModel->price]) ?
                                $data[$itemModel->price] : 0;

                            // Sum values if field value is an array
                            $quantity = is_array($quantity) ? array_sum($quantity) : $quantity;
                            $price = is_array($price) ? array_sum($price) : $price;

                            $amount += $quantity * $price * 100; // Adyen accepts Charges in Cents;
                        }

                        if (isset($state['encryptedCardNumber'], $state['encryptedExpiryMonth'],
                            $state['encryptedExpiryYear'], $state['encryptedSecurityCode'])) {

                            $client = new Client();
                            $client->setApplicationName(Yii::$app->name);
                            $client->setXApiKey($model->api_key);
                            if ($model->live) {
                                $client->setEnvironment(\Adyen\Environment::LIVE, $model->live_url_prefix); //Live environment
                            } else {
                                $client->setEnvironment(Environment::TEST);
                            }
                            $service = new Checkout($client);
                            $params = array(
                                "amount" => array(
                                    "currency" => $model->currency,
                                    "value" => $amount
                                ),
                                "reference" => Yii::$app->name .' - Form #' . $formModel->id . ' - Submission #' . $submissionModel->id,
                                "paymentMethod" => array(
                                    "type" => "scheme",
                                    "encryptedCardNumber" => $state['encryptedCardNumber'],
                                    "encryptedExpiryMonth" => $state['encryptedExpiryMonth'],
                                    "encryptedExpiryYear" => $state['encryptedExpiryYear'],
                                    "encryptedSecurityCode" => $state['encryptedSecurityCode']
                                ),
                                "merchantAccount" => $model->merchant_account
                            );
                            if (isset($state['holderName']) && !empty($state['holderName'])) {
                                $params['paymentMethod']['holderName'] = $state['holderName'];
                            }
                            $result = $service->payments($params);
                            // Append Payment Status to Submission Data
                            if (isset($result['pspReference'], $result['resultCode'])) {
                                // Save payment
                                $payment = new AdyenPayment();
                                $payment->form_id = $model->form->id;
                                $payment->adyen_id = $model->id;
                                $payment->submission_id = $submissionModel->id;
                                $payment->transaction = $result['pspReference'];
                                $payment->result_code = $result['resultCode'];
                                $payment->amount = $amount;
                                $payment->currency = $model->currency;
                                $payment->response = Json::encode($result);
                                $payment->save(false);
                                // Refused transaction
                                if (isset($result['refusalReason'], $result['refusalReasonCode'])) {
                                    throw new Exception($result['refusalReason'], $result['refusalReasonCode']);
                                }
                                // Accepted transaction
                                $data['adyen_psp_reference'] = $result['pspReference'];
                                $data['adyen_result_code'] = $result['resultCode'];
                                $submissionModel->data = $data;
                                // Append Status Token to Form Response
                                if (isset(Yii::$app->params['Form.Response'])) {
                                    $response = Yii::$app->params['Form.Response'];
                                    $response['addons']['adyen'] = [
                                        'adyen_psp_reference' => $result['pspReference'],
                                        'adyen_result_code' => $result['resultCode'],
                                    ];
                                    Yii::$app->params['Form.Response'] = $response;
                                }
                            }
                        }
                    }

                } catch (Exception $e) {

                    // Log exception
                    Yii::error($e);

                }
            }
        }

        return $result;

    }

}