<?php

namespace app\modules\addons\modules\active_campaign\controllers;

use app\components\User;
use Yii;
use Exception;
use yii\filters\VerbFilter;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use app\models\Form;
use app\helpers\ArrayHelper;
use app\modules\addons\modules\active_campaign\models\ActiveCampaign;
use app\modules\addons\modules\active_campaign\models\ActiveCampaignField;
use app\modules\addons\modules\active_campaign\models\ActiveCampaignSearch;
use app\modules\addons\modules\active_campaign\services\ActiveCampaignService;
use yii\filters\AccessControl;

/**
 * AdminController implements the CRUD actions for ActiveCampaign model.
 */
class AdminController extends Controller
{
    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['post'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('ActiveCampaign')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = ActiveCampaign::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\active_campaign\models\ActiveCampaign',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'active-campaign-fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($id, $params) {
                    $formID = isset($params[0]) ? $params[0] : 0;
                    $apiKey = null;
                    $apiUrl = null;
                    $model = ActiveCampaign::findOne(['form_id' => $formID, 'contact_list' => $id]);

                    if (!empty($model->form)) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {
                            $apiKey = $model->api_key;
                            $apiUrl = $model->api_url;
                        }

                    } else {
                        $session = Yii::$app->session;
                        $apiUrl = $session['active_campaign_api_url'];
                        $apiKey = $session['active_campaign_api_key'];
                    }

                    $service = new ActiveCampaignService($apiUrl, $apiKey);
                    return $service->getCustomFields();
                },
                'selectedCallback' => function ($listID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all ActiveCampaign models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new ActiveCampaignSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single ActiveCampaign model.
     * @param integer $id
     * @return mixed
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new ActiveCampaign model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {
        $lists = [];
        $fields = [];
        $connected = false;

        $request = Yii::$app->request;
        $session = Yii::$app->session;

        $model = new ActiveCampaign();

        if ($request->isPost && $model->load($request->post())) {

            try {

                $session['active_campaign_api_url'] = $model->api_url;
                $session['active_campaign_api_key'] = $model->api_key;
                $service = new ActiveCampaignService($model->api_url, $model->api_key);
                $lists = $service->getSubscriberLists();
                $fields = $service->getCustomFields();
                $connected = true;

                if ($model->subscriber_email && $model->validate()) {
                    $model->items = Yii::$app->request->post('ActiveCampaignField',[]);
                    $model->save(false);
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The sign up form has been successfully configured.')
                    );
                    return $this->redirect(['index']);
                }

            } catch (Exception $e) {

                // Log
                Yii::error($e);

                // Show error message
                Yii::$app->getSession()->setFlash(
                    'danger',
                    $e->getMessage()
                );

            }

        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'fieldModel' => new ActiveCampaignField(),
            'forms' => $forms,
            'lists' => $lists,
            'fields' => $fields,
            'connected' => $connected,
        ]);
    }

    /**
     * Updates an existing ActiveCampaign model.
     * If update is successful, the browser will be redirected to the 'view' page.
     *
     * @param $id
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionUpdate($id)
    {
        $model = $this->findModel($id);
        $request = Yii::$app->request;

        if ($model->load($request->post()) && $model->validate()) {
            $model->items = $request->post('ActiveCampaignField',[]);
            $model->save();
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The sign up form has been successfully updated.')
            );
            return $this->redirect(['index']);
        }

        $fieldModel = new ActiveCampaignField();
        $connected = false;

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');
        $lists = [];
        $fields = [];

        try {

            $service = new ActiveCampaignService($model->api_url, $model->api_key);
            $lists = $service->getSubscriberLists();
            $fields = $service->getCustomFields();
            $connected = true;

        } catch(Exception $e) {
            // Log
            Yii::error($e);
            // Show error message
            Yii::$app->getSession()->setFlash('danger', $e->getMessage());
        }

        return $this->render('update', [
            'model' => $model,
            'fieldModel' => $fieldModel,
            'forms' => $forms,
            'lists' => $lists,
            'fields' => $fields,
            'connected' => $connected,
        ]);
    }

    /**
     * Enable / Disable multiple ActiveCampaign models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {
        $models = ActiveCampaign::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        } else {
            foreach ($models as $model) {
                $model->status = $status;
                $model->update();
            }
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The selected items have been successfully updated.')
            );
            return $this->redirect(['index']);
        }
    }

    /**
     * Deletes an existing ActiveCampaign model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Finds the ActiveCampaign model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return ActiveCampaign the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = ActiveCampaign::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
