<?php

namespace app\modules\addons\modules\acellemail\services;

use Exception;
use yii\helpers\Json;

class AcelleMailService
{

    /** @var string Api Key */
    protected $apiKey;

    /** @var string Base Url */
    protected $baseUrl = "https://demo.acellemail.com/api/v1";

    /** @var string Response Format */
    protected $returnFormat = "json";

    /** @var bool Check Header */
    protected $checkHeader = true;

    /** @var bool Throw Exceptions */
    protected $throwExceptions = true;

    /** @var bool Header */
    protected $header = false;

    /** @var bool Error */
    protected $error = false;

    /**
     * AcelleMailService constructor.
     * @param $apiKey
     * @param $baseUrl
     * @throws Exception
     */
    public function __construct($apiKey, $baseUrl)
    {
        if (empty($apiKey) || empty($baseUrl)) {
            throw new Exception('The api key or api url is empty');
        }
        $this->apiKey = $apiKey;
        $this->baseUrl = $baseUrl;
    }

    /**
     * Add or Edit Subscriber
     *
     * @param int $groupId
     * @param array $contact
     * @return array|false
     * @throws Exception
     */
    public function syncContact($listId, $contact)
    {
        $contact['api_token'] = $this->apiKey;
        $contact = (is_array($contact)) ? Json::encode($contact) : $contact;
        $options = [
            CURLOPT_URL => $this->baseUrl . "/subscribers?list_uid={$listId}",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HEADER => false,
            CURLOPT_HTTPHEADER => array(
                "cache-control: no-cache",
                "content-type: application/json"
            ),
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => $contact,
            CURLINFO_HEADER_OUT => true,
        ];
        return (array) $this->request($options);
    }

    /**
     * Get Subscriber Lists
     *
     * @return array
     * @throws Exception
     */
    public function getSubscriberLists()
    {
        $options = [
            CURLOPT_URL => $this->baseUrl . "/lists?api_token={$this->apiKey}",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 0,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HEADER => false,
            CURLOPT_HTTPHEADER => array(
                "cache-control: no-cache",
                "content-type: application/json"
            ),
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLINFO_HEADER_OUT => true,
        ];

        $listCollection  =  (array) $this->request($options);

        $lists = [];
        if (is_array($listCollection) && !empty($listCollection)) {
            foreach($listCollection as $list) {
                $list = (array) $list;
                if (isset($list['uid'], $list['name'])) {
                    $lists[$list['uid']] = $list['name'];
                }
            }
        }

        return $lists;
    }

    /**
     * Get Custom Fields
     *
     * @param int $listId
     * @return array
     * @throws Exception
     */
    public function getCustomFields($listId)
    {
        $options = [
            CURLOPT_URL => $this->baseUrl . "/lists/{$listId}?api_token={$this->apiKey}",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HEADER => false,
            CURLOPT_HTTPHEADER => array(
                "cache-control: no-cache",
                "content-type: application/json"
            ),
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLINFO_HEADER_OUT => true,
        ];

        $response = (array) $this->request($options);
        $fields = [];
        if (isset($response["list"])) {
            $list = (array) $response["list"];
            if (isset($list{"fields"})) {
                $customFieldCollection = (array) $list["fields"];
                if (is_array($customFieldCollection) && !empty($customFieldCollection)) {
                    foreach($customFieldCollection as $customField) {
                        $customField = (array) $customField;
                        if (isset($customField['key'], $customField['label']) && !($customField['key'] === 'EMAIL')) {
                            $fields[$customField['key']] = $customField['label'];
                        }
                    }
                }
            }
        }
        return $fields;
    }

    /**
     * Return formatted response according to settings
     *
     * @param $in
     * @param bool $header
     * @return mixed|null
     * @throws Exception
     */
    protected function response($in, $header = false)
    {
        $this->header = $header;

        if ($this->checkHeader && isset($header["http_code"])) {
            if ($header["http_code"] < 200 || $header["http_code"] >= 300) {
                //error!?
                $this->error = $in;
                $message = var_export($in, true);
                if ($tmp = json_decode($in)) {
                    if (isset($tmp->message)) {
                        $message = $tmp->message;
                    } elseif (isset($tmp->error->message)) {
                        $message = $tmp->error->message;
                    }
                }
                if ($this->throwExceptions) {
                    throw new \Exception(''. $message, $header["http_code"]);
                }
                $in = null;

            }

        }

        switch ($this->returnFormat) {
            case 'json':
                return json_decode($in);
                break;

            default:
                return $in;
                break;
        }
        return $in;
    }

    /**
     * @param $options
     * @return mixed|null
     * @throws Exception
     */
    protected function request($options)
    {
        $curl = curl_init();
        curl_setopt_array($curl, $options);
        $response = curl_exec($curl);
        $info =  curl_getinfo($curl);
        curl_close($curl);

        return $this->response($response, $info);
    }

}