<?php

namespace app\modules\addons\modules\acellemail\controllers;

use app\components\User;
use app\helpers\ArrayHelper;
use app\models\Form;
use app\modules\addons\modules\acellemail\models\AcelleMail;
use app\modules\addons\modules\acellemail\models\AcelleMailField;
use app\modules\addons\modules\acellemail\models\AcelleMailSearch;
use app\modules\addons\modules\acellemail\services\AcelleMailService;
use Exception;
use Yii;
use yii\filters\AccessControl;
use yii\filters\VerbFilter;
use yii\web\Controller;
use yii\web\NotFoundHttpException;

/**
 * AdminController implements the CRUD actions for AcelleMail model.
 */
class AdminController extends Controller
{
    /**
     * @inheritdoc
     */
    public function behaviors()
    {
        return [
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'delete' => ['post'],
                ],
            ],
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['index', 'fields', 'list-fields'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        return ['listing' => true];
                    }],
                    ['actions' => ['create'], 'allow' => true, 'matchCallback' => function ($rule, $action) {
                        if (Yii::$app->request->isGet && Yii::$app->user->can('configureFormsWithAddons', ['listing' => true])) {
                            return true;
                        } else if ($postData = Yii::$app->request->post('AcelleMail')) {
                            if (isset($postData['form_id'])) {
                                if (is_array($postData['form_id'])) {
                                    return ['modelClass' => Form::class, 'ids' => $postData['form_id']];
                                } else {
                                    return ['model' => Form::findOne(['id' => $postData['form_id']])];
                                }
                            } else {
                                return true; // Allow access. This request is not saving data.
                            }
                        }
                        return false;
                    }],
                    ['actions' => ['view', 'update', 'delete'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $model = $this->findModel(Yii::$app->request->get('id'));
                        return ['model' => $model->form];
                    }],
                    ['actions' => ['update-status', 'delete-multiple'], 'allow' => true, 'roles' => ['configureFormsWithAddons'], 'roleParams' => function() {
                        $models = AcelleMail::find()
                            ->where(['in', 'id', Yii::$app->request->post('ids')])
                            ->asArray()->all();
                        $ids = ArrayHelper::getColumn($models, 'form_id');
                        return ['modelClass' => Form::class, 'ids' => $ids];
                    }],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function actions()
    {
        return [
            'delete-multiple' => [
                'class' => 'app\components\actions\DeleteMultipleAction',
                'modelClass' => 'app\modules\addons\modules\acellemail\models\AcelleMail',
                'afterDeleteCallback' => function () {
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The selected items have been successfully deleted.')
                    );
                },
            ],
            'fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($formID, $params) {
                    $output = array();
                    $form = Form::findOne(['id' => $formID]);
                    if ($form) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $form])) {
                            $formDataModel = $form->formData;
                            if ($formDataModel) {
                                $fields = $formDataModel->getFieldsForEmail();
                                foreach ($fields as $name => $label) {
                                    array_push($output, [
                                        'id' => $name,
                                        'name' => $label,
                                    ]);
                                }
                            }
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($formID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ],
            'list-fields' => [
                'class' => \kartik\depdrop\DepDropAction::class,
                'outputCallback' => function ($id, $params) {
                    $output = array();
                    $formID = isset($params[0]) ? $params[0] : 0;
                    $apiKey = null;
                    $apiUrl = null;
                    $model = AcelleMail::findOne(['form_id' => $formID, 'contact_list' => $id]);
                    if (!empty($model->form)) {
                        if (Yii::$app->user->can('configureFormsWithAddons', ['model' => $model->form])) {
                            $apiKey = $model->api_key;
                            $apiUrl = $model->api_url;
                        }
                    } else {
                        $session = Yii::$app->session;
                        $apiKey = $session['acellemail_api_key'];
                        $apiUrl = $session['acellemail_api_url'];
                    }
                    if ($apiKey && $apiUrl) {
                        $service = new AcelleMailService($apiKey, $apiUrl);
                        $fields = $service->getCustomFields($id);
                        foreach ($fields as $name => $label) {
                            array_push($output, [
                                'id' => $name,
                                'name' => $label,
                            ]);
                        }
                    }
                    return $output;
                },
                'selectedCallback' => function ($listID, $params) {
                    if (isset($params[0]) && !empty($params[0])) {
                        return $params[0];
                    }
                }
            ]
        ];
    }

    /**
     * Lists all AcelleMail models.
     * @return mixed
     */
    public function actionIndex()
    {
        $searchModel = new AcelleMailSearch();
        $dataProvider = $searchModel->search(Yii::$app->request->queryParams);

        return $this->render('index', [
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Displays a single AcelleMail model.
     * @param $id
     * @return string
     * @throws NotFoundHttpException
     */
    public function actionView($id)
    {
        return $this->render('view', [
            'model' => $this->findModel($id),
        ]);
    }

    /**
     * Creates a new AcelleMail model.
     * If creation is successful, the browser will be redirected to the 'view' page.
     * @return mixed
     */
    public function actionCreate()
    {

        $lists = [];
        $connected = false;

        $model = new AcelleMail();
        $fieldModel = new AcelleMailField();

        $request = Yii::$app->request;

        if ($request->isPost && $model->load($request->post())) {

            try {

                $session = Yii::$app->session;
                $session['acellemail_api_key'] = $model->api_key;
                $session['acellemail_api_url'] = $model->api_url;

                $service = new AcelleMailService($model->api_key, $model->api_url);
                $lists = $service->getSubscriberLists();
                $connected = true;

                if ($model->subscriber_email && $model->validate()) {
                    $model->items = Yii::$app->request->post('AcelleMailField',[]);
                    $model->save(false);
                    Yii::$app->getSession()->setFlash(
                        'success',
                        Yii::t('app', 'The sign up form has been successfully configured.')
                    );
                    return $this->redirect(['index']);
                }

            } catch (Exception $e) {

                // Log
                Yii::error($e);

                // Show error message
                Yii::$app->getSession()->setFlash(
                    'danger',
                    $e->getMessage()
                );

            }
        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('create', [
            'model' => $model,
            'fieldModel' => $fieldModel,
            'forms' => $forms,
            'lists' => $lists,
            'connected' => $connected,
        ]);
    }

    /**
     * Updates an existing AcelleMail model.
     * If update is successful, the browser will be redirected to the 'view' page.
     * @param $id
     * @return string|\yii\web\Response
     * @throws NotFoundHttpException
     */
    public function actionUpdate($id)
    {

        $model = $this->findModel($id);
        $request = Yii::$app->request;

        if ($model->load($request->post()) && $model->validate()) {
            $items = Yii::$app->request->post('AcelleMailField',[]);
            if (!empty($items)) {
                $model->items = $items;
            }
            $model->save();
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The sign up form has been successfully updated.')
            );
            return $this->redirect(['index']);
        }

        // Initial values
        $connected = false;
        $lists = [];

        try {

            $service = new AcelleMailService($model->api_key, $model->api_url);
            $lists = $service->getSubscriberLists();
            $connected = true;

        } catch(Exception $e) {

            // Log
            Yii::error($e);

            // Show error message
            Yii::$app->getSession()->setFlash(
                'danger',
                $e->getMessage()
            );

        }

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;
        $forms = $currentUser->forms()->orderBy('updated_at DESC')->asArray()->all();
        $forms = ArrayHelper::map($forms, 'id', 'name');

        return $this->render('update', [
            'model' => $model,
            'fieldModel' => new AcelleMailField(),
            'forms' => $forms,
            'lists' => $lists,
            'connected' => $connected,
        ]);
    }

    /**
     * Enable / Disable multiple CleverReach models
     *
     * @param $status
     * @return \yii\web\Response
     * @throws NotFoundHttpException
     * @throws \Exception
     */
    public function actionUpdateStatus($status)
    {
        $models = AcelleMail::findAll(['id' => Yii::$app->getRequest()->post('ids')]);

        if (empty($models)) {
            throw new NotFoundHttpException(Yii::t('app', 'Page not found.'));
        } else {
            foreach ($models as $model) {
                $model->status = $status;
                $model->update();
            }
            Yii::$app->getSession()->setFlash(
                'success',
                Yii::t('app', 'The selected items have been successfully updated.')
            );
            return $this->redirect(['index']);
        }
    }

    /**
     * Deletes an existing AcelleMail model.
     * If deletion is successful, the browser will be redirected to the 'index' page.
     * @param integer $id
     * @return mixed
     */
    public function actionDelete($id)
    {
        $this->findModel($id)->delete();

        return $this->redirect(['index']);
    }

    /**
     * Finds the AcelleMail model based on its primary key value.
     * If the model is not found, a 404 HTTP exception will be thrown.
     * @param integer $id
     * @return AcelleMail the loaded model
     * @throws NotFoundHttpException if the model cannot be found
     */
    protected function findModel($id)
    {
        if (($model = AcelleMail::findOne(['id' => $id])) !== null) {
            return $model;
        } else {
            throw new NotFoundHttpException('The requested page does not exist.');
        }
    }
}
