<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.1
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\helpers;

use app\components\gmail\GmailService;
use app\components\queue\MailQueue;
use app\components\queue\Message;
use app\components\Settings;
use app\controllers\FormController;
use app\events\SubmissionMailEvent;
use app\models\Form;
use app\models\FormSubmission;
use app\modules\addons\modules\stripe\models\StripePayment;
use app\modules\addons\modules\stripe\models\StripePaymentSearch;
use Google\Client;
use Google\Service\Gmail;
use Google_Service_Gmail_Message;
use PHPMailer\PHPMailer\PHPMailer;
use Swift_Attachment;
use Yii;

/**
 * Class Mailer
 * @package app\helpers
 *
 * Business logic to send emails
 */
class MailHelper
{

    /**
     * Return the sender email address according to app configuration
     *
     * @param string $sender Email by default
     * @return string Email address
     */
    public static function from($sender = '')
    {

        /** @var MailQueue $mailer */
        $mailer = Yii::$app->mailer;
        /** @var Settings $settings */
        $settings = Yii::$app->settings;

        // Check for messageConfig before sending (for backwards-compatible purposes)
        if (isset($mailer->messageConfig, $mailer->messageConfig["from"]) &&
            !empty($mailer->messageConfig["from"])) {
            $sender = $mailer->messageConfig["from"];
        } if (isset(Yii::$app->params['App.Modules.SMTP.username']) &&
            !empty(Yii::$app->params['App.Modules.SMTP.username']) &&
            (!filter_var(Yii::$app->params['App.Modules.SMTP.username'], FILTER_VALIDATE_EMAIL) === false)) {
            $sender = Yii::$app->params['App.Modules.SMTP.username'];
        } elseif (isset(Yii::$app->params['App.Mailer.transport']) &&
            Yii::$app->params['App.Mailer.transport'] === 'smtp' &&
            (!filter_var($settings->get("smtp.username"), FILTER_VALIDATE_EMAIL) === false)) {
            // Set smtp username as sender
            $sender = $settings->get("smtp.username");
        }

        // Add name to Sender
        if (is_string($sender)
            && (!filter_var($sender, FILTER_VALIDATE_EMAIL) === false)) {
            $from = $sender;
            if (!empty(Yii::$app->params['App.Modules.SMTP.fromName'])
                && $senderName = trim(Yii::$app->params['App.Modules.SMTP.fromName']) ?? '') {
                $from = [$sender => $senderName];
            } elseif ($senderName = trim($settings->get("app.defaultFromName") ?? '')) {
                $from = [$sender => $senderName];
            } elseif ($senderName = trim($settings->get("app.name") ?? '')) {
                $from = [$sender => $senderName];
            }
            $sender = $from;
        }

        return $sender;
    }

    /**
     * Check if the email should be asynchronous
     *
     * @return bool
     */
    public static function async()
    {
        /** @var Settings $settings */
        $settings = Yii::$app->settings;

        // Async Email
        $async = !empty(Yii::$app->params['App.Mailer.async']) && Yii::$app->params['App.Mailer.async'] === 1;
        $async = (boolean) $settings->get('async', 'app', $async);
        $async = $async && $settings->get('mailerTransport', 'app', 'php') === 'smtp';
        return $async;
    }

    /**
     * Send Notification Message By E-Mail
     *
     * @param Form $formModel
     * @param FormSubmission $submissionModel
     * @param array $filePaths
     * @return bool
     * @throws \Exception
     */
    public static function sendNotificationByEmail($formModel, $submissionModel, $filePaths = [])
    {
        $result = false;
        // Each email address should be unique between to, cc, and bcc
        $toCcBcc = [];

        /** @var Settings $settings */
        $settings = Yii::$app->settings;

        $dataModel = $formModel->formData;
        $emailModel = $formModel->formEmail;
        $submissionData = $submissionModel->getSubmissionData();

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getUrl();
            }
        }

        $mailTo = [];

        // Recipient (Email address)
        // FormEmail[to]
        // $emailModel->to

        // Recipient (Email field)
        // FormEmail[field_to]
        // $emailModel->field_to

        if (!empty($emailModel->to)) {
            $emails = explode(',', $emailModel->to);
            foreach ($emails as $email) {
                if (!in_array($email, $toCcBcc)) {
                    $mailTo[] = $email;
                    $toCcBcc[] = $email;
                }
            }
        }

        if (!empty($emailModel->field_to)) {
            foreach ($emailModel->field_to as $fieldTo) {
                // To (Get value of email field)
                if (!empty($submissionData) && !empty($submissionData[$fieldTo]) && is_array($submissionData[$fieldTo]))
                    $to = $submissionData[$fieldTo][0] ?? null;
                else
                    $to = $submissionData[$fieldTo] ?? null;
                // Remove all illegal characters from email
                $to = filter_var($to, FILTER_SANITIZE_EMAIL);
                // Validate e-mail
                if (!filter_var($to, FILTER_VALIDATE_EMAIL) === false) {
                    if (!in_array($to, $toCcBcc)) {
                        $mailTo[] = $to;
                        $toCcBcc[] = $to;
                    }
                }
            }
        }

        // Check first: Recipient and Sender are required
        if (isset($emailModel->from) && !empty($mailTo)) {

            // Async Email
            $async = static::async();

            // Sender by default: No-Reply Email
            $fromEmail = static::from($settings->get("app.noreplyEmail"));
            // Sender verification
            if (empty($fromEmail)) {
                return false;
            }

            // Form fields
            $fieldsForEmail = $dataModel->getFieldsForEmail();
            // Submission data in an associative array
            $tokens = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
            // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
            $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);

            // Data
            $data = [
                'fieldValues' => $tokens, // Submission data for replacement
                'fieldData' => $fieldData, // Submission data for print details
                'formID' => $formModel->id,
                'submissionID' => isset($submissionModel->primaryKey) ? $submissionModel->primaryKey : null,
                'submissionModel' => $submissionModel,
                'message' => $emailModel->message,
                'receipt_copy' => (boolean) $emailModel->receipt_copy, // Add submission copy
            ];

            // Views
            $notificationViews = $emailModel->getEmailViews();
            // Subject
            $subject = isset($emailModel->subject) && !empty($emailModel->subject) ?
                $emailModel->subject :
                $formModel->name . ' - ' . Yii::t('app', 'New Submission');
            // Token replacement in subject
            $subject = SubmissionHelper::replaceTokens($subject, $tokens);
            $subject = Liquid::render($subject, $tokens);
            // When a submission is updated
            if (!empty($submissionModel->updated_at) &&
                $submissionModel->updated_at > $submissionModel->created_at) {
                $subject = $subject . ' - ' . Yii::t('app', 'Edited');
            }

            // Add name to From
            if (!empty($emailModel->from_name)) {
                $fromName = isset($submissionData[$emailModel->from_name]) ?
                    $submissionData[$emailModel->from_name] : $emailModel->from_name;
                if (!empty($fromName)) {
                    $fromName = is_array($fromName) ? implode(",", $fromName) : $fromName;
                    if (is_array($fromEmail)) {
                        $fromEmail = [
                            key($fromEmail) => $fromName,
                        ];
                    } else {
                        $fromEmail = [
                            $fromEmail => $fromName,
                        ];
                    }
                }
            }

            // Compose email
            /** @var Message $message */
            $message = Yii::$app->mailer->compose($notificationViews, $data)
                ->setFrom($fromEmail)
                ->setTo($mailTo)
                ->setSubject($subject);

            // Reply to
            if (!empty($emailModel->from)) {
                // ReplyTo (can be an email or an email field)
                $replyToField = isset($submissionData[$emailModel->from]) ? $submissionData[$emailModel->from] : null;
                $replyTo = $emailModel->fromIsEmail() ? $emailModel->from :
                    $replyToField;

                // Add name to Reply to
                if (!empty($replyTo) && !empty($emailModel->from_name)) {
                    $fromName = isset($submissionData[$emailModel->from_name]) ?
                        $submissionData[$emailModel->from_name] : $emailModel->from_name;
                    $fromName = is_array($fromName) ? implode(",", $fromName) : $fromName;

                    if (is_array($replyTo)) {
                        $replyTo = [
                            key($replyTo) => $fromName,
                        ];
                    } else {
                        $replyTo = [
                            $replyTo => $fromName,
                        ];
                    }
                }

                if (!empty($replyTo)) {
                    $message->setReplyTo($replyTo);
                }
            }

            // Add cc
            if (is_array($emailModel->cc)) {
                foreach ($emailModel->cc as $cc) {
                    if (!empty($submissionData[$cc])) {

                        // An email field can have multiple emails
                        $emails = explode(',', $submissionData[$cc]);
                        foreach ($emails as $email) {
                            if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
                                if (!in_array($email, $toCcBcc)) {
                                    $message->setCc($email);
                                    $toCcBcc[] = $email;
                                }
                            }
                        }
                    } elseif (!filter_var($cc, FILTER_VALIDATE_EMAIL) === false) {
                        if (!in_array($cc, $toCcBcc)) {
                            $message->setCc($cc);
                            $toCcBcc[] = $cc;
                        }
                    }
                }
            }

            // Add bcc
            if (is_array($emailModel->bcc)) {
                foreach ($emailModel->bcc as $bcc) {
                    if (!empty($submissionData[$bcc])) {
                        // An email field can have multiple emails
                        $emails = explode(',', $submissionData[$bcc]);
                        foreach ($emails as $email) {
                            if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
                                if (!in_array($email, $toCcBcc)) {
                                    $message->setBcc($email);
                                    $toCcBcc[] = $email;
                                }
                            }
                        }
                    } elseif (!filter_var($bcc, FILTER_VALIDATE_EMAIL) === false) {
                        if (!in_array($bcc, $toCcBcc)) {
                            $message->setBcc($bcc);
                            $toCcBcc[] = $bcc;
                        }
                    }
                }
            }

            // Attach files
            if ($emailModel->attach && count($filePaths) > 0) {
                foreach ($filePaths as $filePath) {
                    $message->attach(Yii::getAlias('@app') . DIRECTORY_SEPARATOR . $filePath);
                }
            }

            // Trigger Event
            Yii::$app->trigger(SubmissionMailEvent::EVENT_NAME, new SubmissionMailEvent([
                'submission' => $submissionModel,
                'type' => SubmissionMailEvent::EVENT_TYPE_NOTIFICATION,
                'message' => $message,
                'async' => $async,
                'tokens' => $tokens,
            ]));

            // Send E-mail
            if ($async) {
                $result = $message->queue();
            } else {
                $toEmail = implode(',',array_keys($message->getTo()));
                $subject = $message->getSubject();

                $replyTo = '';
                $cc = '';
                $bcc = '';

                if(!empty($message->getReplyTo()) && is_array($message->getReplyTo()))
                    $replyTo = implode(',',array_keys($message->getReplyTo()));
                if(!empty($message->getCc()) && is_array($message->getCc()))
                    $cc = implode(',',array_keys($message->getCc()));
                if(!empty($message->getBcc()) && is_array($message->getBcc()))
                    $bcc = implode(',',array_keys($message->getBcc()));

                Yii::$app->controller->layout = false;
                $output = Yii::$app->controller->render('@app/mail/submission-html', $data);
                $body = Yii::$app->controller->render('@app/mail/layouts/html', ['content' => $output, 'message' => '']);

                $service = new GmailService();
//                $msg = $service->createMessage($toEmail, $replyTo, $cc, $bcc, $subject, $body);
////                throw new \Exception($msg->getRaw());
//                $result = $service->sendMessage($msg);


                $service = new GmailService();
                $result = $service->sendMessage($service->createMessage($fromEmail, $message->getTo(), $message->getReplyTo(), $message->getCc(), $message->getBcc(), $subject, $body));

//                $result = $message->send();
            }

        }

        return $result;
    }

    /**
     * Send Confirmation Message By E-Mail
     *
     * @param Form $formModel
     * @param FormSubmission $submissionModel
     * @param array $filePaths
     * @return bool
     * @throws \Exception
     */
    public static function sendConfirmationByEmail($formModel, $submissionModel, $filePaths = [])
    {

        $result = false;
        // Each email address should be unique between to, cc, and bcc
        $toCcBcc = [];

        /** @var Settings $settings */
        $settings = Yii::$app->settings;

        $dataModel = $formModel->formData;
        $confirmationModel = $formModel->formConfirmation;
        $submissionData = $submissionModel->getSubmissionData();
        $payments = StripePaymentSearch::findAll(['submission_id' => $submissionModel->id, 'status' => 'Complete']);

        // If file paths are empty, find them by model relation
        if (empty($filePaths)) {
            $fileModels = $submissionModel->files;
            foreach ($fileModels as $fileModel) {
                $filePaths[] = $fileModel->getUrl();
            }
        }

        // Check first: Send email must be active and Recipient is required
        if ($confirmationModel->send_email && !empty($confirmationModel->mail_to)) {

            // Async Email
            $async = static::async();

            // Sender by default: No-Reply Email
            $fromEmail = static::from($settings->get("app.noreplyEmail"));
            // Sender verification
            if (empty($fromEmail)) {
                return false;
            }

            // Form fields
            $fieldsForEmail = $dataModel->getFieldsForEmail();
            // Submission data in an associative array
            $tokens = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
            // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
            $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);

            $mailTo = [];

            foreach ($confirmationModel->mail_to as $fieldTo) {
                // To (Get value of email field)
                $to = $submissionData[$fieldTo] ?? null;
                // Remove all illegal characters from email
                $to = filter_var($to, FILTER_SANITIZE_EMAIL);
                // Validate e-mail
                if (!filter_var($to, FILTER_VALIDATE_EMAIL) === false) {
                    if (!in_array($to, $toCcBcc)) {
                        $mailTo[] = $to;
                        $toCcBcc[] = $to;
                    }
                }
            }

            if (!empty($mailTo)) {

                // Views
                $confirmationViews = $confirmationModel->getEmailViews();

                // Message
                $data = [
                    'fieldValues' => $tokens, // Submission data for replacement
                    'fieldData' => $fieldData, // Submission data for print details
                    'mail_receipt_copy' => (boolean) $confirmationModel->mail_receipt_copy, // Add submission copy
                    'message' => !empty($confirmationModel->mail_message)
                        ? $confirmationModel->mail_message
                        : Yii::t('app', 'Thanks for your message'),
                    'payment_type' => $submissionModel->payment_type,
                    'manual_payment_link' => FormController::generateManualCheckout($formModel->created_by,$submissionModel->id)
                ];

                // Subject
                $subject = !empty($confirmationModel->mail_subject)
                    ? $confirmationModel->mail_subject
                    : $formModel->name . ' - Confirmation';

                // Token replacement in subject
                $subject = SubmissionHelper::replaceTokens($subject, $tokens);
                $subject = Liquid::render($subject, $tokens);

                // Add name to From
                if (!empty($confirmationModel->mail_from_name)) {
                    if (is_array($fromEmail)) {
                        $fromEmail = [
                            key($fromEmail) => $confirmationModel->mail_from_name,
                        ];
                    } else {
                        $fromEmail = [
                            $fromEmail => $confirmationModel->mail_from_name,
                        ];
                    }
                }

                // Compose email
                /** @var Message $message */
                $message = Yii::$app->mailer->compose($confirmationViews, $data)
                    ->setFrom($fromEmail)
                    ->setTo($mailTo)
                    ->setSubject($subject);

                // Add reply to
                if (!empty($confirmationModel->mail_from)) {
                    $message->setReplyTo($confirmationModel->mail_from);
                }

                // Add cc
                if (!empty($confirmationModel->mail_cc)) {
                    $mailCc = explode(',', $confirmationModel->mail_cc);
                    $emails = [];
                    foreach ($mailCc as $email) {
                        if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
                            if (!in_array($email, $toCcBcc)) {
                                $emails[] = $email;
                                $toCcBcc[] = $email;
                            }
                        }
                    }
                    $message->setCc($emails);
                }

                // Add bcc
                if (!empty($confirmationModel->mail_bcc)) {
                    $mailBcc = explode(',', $confirmationModel->mail_bcc);
                    $emails = [];
                    foreach ($mailBcc as $email) {
                        if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
                            if (!in_array($email, $toCcBcc)) {
                                $emails[] = $email;
                                $toCcBcc[] = $email;
                            }
                        }
                    }
                    $message->setBcc($emails);
                }

                // Attach files
                if ($confirmationModel->mail_attach && count($filePaths) > 0) {
                    foreach ($filePaths as $filePath) {
                        $message->attach(Yii::getAlias('@app') . DIRECTORY_SEPARATOR . $filePath);
                    }
                }

                // Trigger Event
                Yii::$app->trigger(SubmissionMailEvent::EVENT_NAME, new SubmissionMailEvent([
                    'submission' => $submissionModel,
                    'type' => SubmissionMailEvent::EVENT_TYPE_CONFIRMATION,
                    'message' => $message,
                    'async' => $async,
                    'tokens' => $tokens,
                ]));

                // Send E-mail
                if ($async) {
                    $result = $message->queue();
                } else {
                    $toEmail = implode(',',array_keys($message->getTo()));
                    $subject = $message->getSubject();

                    $replyTo = '';
                    $cc = '';
                    $bcc = '';

                    if(!empty($message->getReplyTo()) && is_array($message->getReplyTo()))
                        $replyTo = implode(',',array_keys($message->getReplyTo()));
                    if(!empty($message->getCc()) && is_array($message->getCc()))
                        $cc = implode(',',array_keys($message->getCc()));
                    if(!empty($message->getBcc()) && is_array($message->getBcc()))
                        $bcc = implode(',',array_keys($message->getBcc()));

                    Yii::$app->controller->layout = false;
                    $output = Yii::$app->controller->render('@app/mail/confirmation-html', $data);
                    $body = Yii::$app->controller->render('@app/mail/layouts/html', ['content' => $output, 'message' => '']);

                    $attachment = null;
//                    echo $submissionModel->id;
//                    print_r($payments);

                    if (count($payments) > 0) {
//                        print_r($payments);
                        $attachment = Swift_Attachment::fromPath(reset($payments)[0]->getReceiptLink() . '&pdf=1')->setFilename('Payment Receipt.pdf');
                    }

                    $service = new GmailService();
                    $result = $service->sendMessage($service->createMessage($fromEmail, $message->getTo(), $message->getReplyTo(), $message->getCc(), $message->getBcc(), $subject, $body, $attachment));

                }
            }
        }

        return $result;
    }

    /**
     * Send Payment Confirmation Message By E-Mail
     *
     * @param StripePayment $paymentModel
     * @return bool
     * @throws \Exception
     */
    public static function sendManualPaymentConfirmationByEmail($paymentModel)
    {
        $result = false;
        // Each email address should be unique between to, cc, and bcc
        $toCcBcc = [];

        /** @var Settings $settings */
        $settings = Yii::$app->settings;


        // Check first: Send email must be active and Recipient is required
        if (!empty($paymentModel->customer_email)) {
            // Async Email
            $async = static::async();

            // Sender by default: No-Reply Email
            $fromEmail = static::from($settings->get("app.noreplyEmail"));
            // Sender verification
            if (empty($fromEmail)) {
                return false;
            }

            $mailTo = [];

            // To (Get value of email field)
            $to = $paymentModel->customer_email;

            // Remove all illegal characters from email
            $to = filter_var($to, FILTER_SANITIZE_EMAIL);
            // Validate e-mail
            if (!filter_var($to, FILTER_VALIDATE_EMAIL) === false) {
                if (!in_array($to, $toCcBcc)) {
                    $mailTo[] = $to;
                    $toCcBcc[] = $to;
                }
            }

            if (!empty($mailTo)) {
                $paymentConfirmationMessage = '
                    <p>Assalamu Alaikum,</p>
                    
                    <p>Thank you for your payment. We are pleased to confirm that we have received your payment for your upcoming journey with Dar El Salam Travel.</p>
                    
                    <p>May Allah accept your payment and make your journey blessed.</p>
                
                    <p><b>A detailed receipt has been attached to this email.</b></p>
                    
                ';

                // Message
                $data = [
                    'fieldValues' => [],
                    'fieldData' => [],
                    'mail_receipt_copy' => false, // Add submission copy
                    'message' => $paymentConfirmationMessage
                ];

                // Subject
                $subject = 'Manual Payment Confirmation';

//                // Send E-mail
//                if ($async) {
//                    $result = $message->queue();
//                } else {
                    $output = Yii::$app->controller->render('@app/mail/payment-confirmation-html', $data);
                    $body = Yii::$app->controller->render('@app/mail/layouts/html', ['content' => $output, 'message' => '']);

                    $attachment = Swift_Attachment::fromPath($paymentModel->getReceiptLink() . '&pdf=1')->setFilename('Payment Receipt.pdf');

                    $service = new GmailService();
                    $result = $service->sendMessage($service->createMessage($fromEmail, $mailTo, [], [], [], $subject, $body, $attachment));
//                }
            }
        }

        return $result;
    }


    /**
     * Send Payment Confirmation Message By E-Mail
     *
     * @param StripePayment $paymentModel
     * @return bool
     * @throws \Exception
     */
    public static function sendPaymentConfirmationByEmail($paymentModel)
    {
        $result = false;
        // Each email address should be unique between to, cc, and bcc
        $toCcBcc = [];

        /** @var Settings $settings */
        $settings = Yii::$app->settings;

        $formModel = $paymentModel->form;
//        $dataModel = $formModel->formData;
        $confirmationModel = $formModel->formConfirmation;
        $submissionData = $paymentModel->submission->getSubmissionData();
//        $payments = $submissionModel->getPayments();

//        // If file paths are empty, find them by model relation
//        if (empty($filePaths)) {
//            $fileModels = $submissionModel->files;
//            foreach ($fileModels as $fileModel) {
//                $filePaths[] = $fileModel->getUrl();
//            }
//        }


        // Check first: Send email must be active and Recipient is required
        if ($confirmationModel->send_email && !empty($confirmationModel->mail_to)) {
            // Async Email
            $async = static::async();

            // Sender by default: No-Reply Email
            $fromEmail = static::from($settings->get("app.noreplyEmail"));
            // Sender verification
            if (empty($fromEmail)) {
                return false;
            }

            // Form fields
//            $fieldsForEmail = $dataModel->getFieldsForEmail();
            // Submission data in an associative array
//            $tokens = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
            // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
//            $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);

            $mailTo = [];

            foreach ($confirmationModel->mail_to as $fieldTo) {
                // To (Get value of email field)
                $to = $submissionData[$fieldTo] ?? null;

                // Remove all illegal characters from email
                $to = filter_var($to, FILTER_SANITIZE_EMAIL);
                // Validate e-mail
                if (!filter_var($to, FILTER_VALIDATE_EMAIL) === false) {
                    if (!in_array($to, $toCcBcc)) {
                        $mailTo[] = $to;
                        $toCcBcc[] = $to;
                    }
                }
            }

            if (!empty($mailTo)) {

                // Views
                $confirmationViews = $confirmationModel->getEmailViews();

                $paymentConfirmationMessage = '
            <p>Assalamu Alaikum,</p>
            
            <p>Thank you for your payment. We are pleased to confirm that we have received your payment for your upcoming journey with Dar El Salam Travel.</p>
            
            <p>May Allah accept your payment and make your journey blessed.</p>
        
            <p><b>A detailed receipt has been attached to this email.</b></p>
            
            ';

                // Message
                $data = [
                    'fieldValues' => [],
                    'fieldData' => [],
//                    'fieldValues' => $tokens, // Submission data for replacement
//                    'fieldData' => $fieldData, // Submission data for print details
                    'mail_receipt_copy' => (boolean) $confirmationModel->mail_receipt_copy, // Add submission copy
                    'message' => $paymentConfirmationMessage
                ];

                // Subject
                $subject = $formModel->name . ' - Payment Confirmation';

                // Add name to From
                if (!empty($confirmationModel->mail_from_name)) {
                    if (is_array($fromEmail)) {
                        $fromEmail = [
                            key($fromEmail) => $confirmationModel->mail_from_name,
                        ];
                    } else {
                        $fromEmail = [
                            $fromEmail => $confirmationModel->mail_from_name,
                        ];
                    }
                }

                // Compose email
                /** @var Message $message */
                $message = Yii::$app->mailer->compose($confirmationViews, $data)
                    ->setFrom($fromEmail)
                    ->setTo($mailTo)
                    ->setSubject($subject);

                // Add reply to
                if (!empty($confirmationModel->mail_from)) {
                    $message->setReplyTo($confirmationModel->mail_from);
                }

                // Add cc
                if (!empty($confirmationModel->mail_cc)) {
                    $mailCc = explode(',', $confirmationModel->mail_cc);
                    $emails = [];
                    foreach ($mailCc as $email) {
                        if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
                            if (!in_array($email, $toCcBcc)) {
                                $emails[] = $email;
                                $toCcBcc[] = $email;
                            }
                        }
                    }
                    $message->setCc($emails);
                }

                // Add bcc
                if (!empty($confirmationModel->mail_bcc)) {
                    $mailBcc = explode(',', $confirmationModel->mail_bcc);
                    $emails = [];
                    foreach ($mailBcc as $email) {
                        if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
                            if (!in_array($email, $toCcBcc)) {
                                $emails[] = $email;
                                $toCcBcc[] = $email;
                            }
                        }
                    }
                    $message->setBcc($emails);
                }

//                // Attach files
//                if ($confirmationModel->mail_attach && count($filePaths) > 0) {
//                    foreach ($filePaths as $filePath) {
//                        $message->attach(Yii::getAlias('@app') . DIRECTORY_SEPARATOR . $filePath);
//                    }
//                }

//                // Trigger Event
//                Yii::$app->trigger(SubmissionMailEvent::EVENT_NAME, new SubmissionMailEvent([
//                    'submission' => $submissionModel,
//                    'type' => SubmissionMailEvent::EVENT_TYPE_CONFIRMATION,
//                    'message' => $message,
//                    'async' => $async,
//                    'tokens' => $tokens,
//                ]));

                // Send E-mail
                if ($async) {
                    $result = $message->queue();
                } else {
                    $toEmail = implode(',',array_keys($message->getTo()));
                    $subject = $message->getSubject();

                    $replyTo = '';
                    $cc = '';
                    $bcc = '';

                    if(!empty($message->getReplyTo()) && is_array($message->getReplyTo()))
                        $replyTo = implode(',',array_keys($message->getReplyTo()));
                    if(!empty($message->getCc()) && is_array($message->getCc()))
                        $cc = implode(',',array_keys($message->getCc()));
                    if(!empty($message->getBcc()) && is_array($message->getBcc()))
                        $bcc = implode(',',array_keys($message->getBcc()));

                    $output = Yii::$app->controller->render('@app/mail/payment-confirmation-html', $data);
                    $body = Yii::$app->controller->render('@app/mail/layouts/html', ['content' => $output, 'message' => '']);

                    $attachment = null;
//                    echo $submissionModel->id;
//                    print_r($payments);

//                    if (count($payments) > 0) {
//                        print_r($payments);
                        $attachment = Swift_Attachment::fromPath($paymentModel->getReceiptLink() . '&pdf=1')->setFilename('Payment Receipt.pdf');
//                    }

                    $service = new GmailService();
                    $result = $service->sendMessage($service->createMessage($fromEmail, $message->getTo(), $message->getReplyTo(), $message->getCc(), $message->getBcc(), $subject, $body, $attachment));
                }
            }
        }

        return $result;
    }


    /**
     * Send Payment Verification Message By E-Mail
     *
     * @param string $verificationLink
     * @param string $mailTo
     * @return bool
     * @throws \Exception
     */
    public static function sendPaymentVerificationByEmail($mailTo, $verificationLink)
    {
        $result = false;
        $toCcBcc = [];

        /** @var Settings $settings */
        $settings = Yii::$app->settings;


//        $confirmationModel = $formModel->formConfirmation;



//        // Check first: Send email must be active and Recipient is required
//        if ($confirmationModel->send_email && !empty($confirmationModel->mail_to)) {
            // Async Email
//            $async = static::async();

            // Sender by default: No-Reply Email
            $fromEmail = static::from($settings->get("app.noreplyEmail"));
            // Sender verification
            if (empty($fromEmail)) {
                return false;
            }

//            $mailTo = [];
//
//            foreach ($confirmationModel->mail_to as $fieldTo) {
//                // To (Get value of email field)
//                $to = $submissionData[$fieldTo] ?? null;
//
//                // Remove all illegal characters from email
//                $to = filter_var($to, FILTER_SANITIZE_EMAIL);
//                // Validate e-mail
//                if (!filter_var($to, FILTER_VALIDATE_EMAIL) === false) {
//                    if (!in_array($to, $toCcBcc)) {
//                        $mailTo[] = $to;
//                        $toCcBcc[] = $to;
//                    }
//                }
//            }

            if (!empty($mailTo)) {

                // Views
//                $confirmationViews = $confirmationModel->getEmailViews();

                $paymentConfirmationMessage = '
            <p>Assalamu Alaikum,</p>
            
            <p>We received a payment from you that requires additional verification in order to be processed.</p>
            
            <p style="font-weight:bold;">You will see a $0.01 deposit in your account in 1-2 business days. Once you receive the deposit in your account, follow this link to verify your account and complete payment:</p>
            
            <p>' . $verificationLink . '</p>
            
            <p>May Allah accept your payment and make your journey blessed.</p>
            
            ';

                // Message
                $data = [
                    'fieldValues' => [],
                    'fieldData' => [],
//                    'mail_receipt_copy' => (boolean) $confirmationModel->mail_receipt_copy, // Add submission copy
                    'message' => $paymentConfirmationMessage
                ];

                // Subject
//                $subject = !empty($confirmationModel->mail_subject)
//                    ? $confirmationModel->mail_subject
//                    : Yii::t('app', 'Thanks for your message');
//                $subject = $formModel->name . ' - Payment Verification';

                // Add name to From
//                if (!empty($confirmationModel->mail_from_name)) {
//                    if (is_array($fromEmail)) {
//                        $fromEmail = [
//                            key($fromEmail) => $confirmationModel->mail_from_name,
//                        ];
//                    } else {
//                        $fromEmail = [
//                            $fromEmail => $confirmationModel->mail_from_name,
//                        ];
//                    }
//                }

                // Compose email
//                /** @var Message $message */
//                $message = Yii::$app->mailer->compose($confirmationViews, $data)
//                    ->setFrom($fromEmail)
//                    ->setTo($mailTo)
//                    ->setSubject($subject);
//
//                // Add reply to
//                if (!empty($confirmationModel->mail_from)) {
//                    $message->setReplyTo($confirmationModel->mail_from);
//                }
//
//                // Add cc
//                if (!empty($confirmationModel->mail_cc)) {
//                    $mailCc = explode(',', $confirmationModel->mail_cc);
//                    $emails = [];
//                    foreach ($mailCc as $email) {
//                        if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
//                            if (!in_array($email, $toCcBcc)) {
//                                $emails[] = $email;
//                                $toCcBcc[] = $email;
//                            }
//                        }
//                    }
//                    $message->setCc($emails);
//                }
//
//                // Add bcc
//                if (!empty($confirmationModel->mail_bcc)) {
//                    $mailBcc = explode(',', $confirmationModel->mail_bcc);
//                    $emails = [];
//                    foreach ($mailBcc as $email) {
//                        if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
//                            if (!in_array($email, $toCcBcc)) {
//                                $emails[] = $email;
//                                $toCcBcc[] = $email;
//                            }
//                        }
//                    }
//                    $message->setBcc($emails);
//                }


                // Send E-mail
//                if ($async) {
//                    $result = $message->queue();
//                } else {
//                    $toEmail = $toEmail;
                    $subject = 'Payment Verification';

                    $replyTo = '';
                    $cc = '';
                    $bcc = '';

//                    if(!empty($message->getReplyTo()) && is_array($message->getReplyTo()))
//                        $replyTo = implode(',',array_keys($message->getReplyTo()));
//                    if(!empty($message->getCc()) && is_array($message->getCc()))
//                        $cc = implode(',',array_keys($message->getCc()));
//                    if(!empty($message->getBcc()) && is_array($message->getBcc()))
//                        $bcc = implode(',',array_keys($message->getBcc()));

                    $output = Yii::$app->controller->render('@app/mail/payment-confirmation-html', $data);
                    $body = Yii::$app->controller->render('@app/mail/layouts/html', ['content' => $output, 'message' => '']);

                    $service = new GmailService();
                    $result = $service->sendMessage($service->createMessage($fromEmail, $mailTo, $replyTo, $cc, $bcc, $subject, $body));
//                }
            }
//        }

        return $result;
    }

    /**
     * Send Failed Payment Message By E-Mail
     *
     * @param string $paymentLink
     * @param string $mailTo
     * @return bool
     * @throws \Exception
     */
    public static function sendPaymentFailureByEmail($mailTo, $paymentLink = null)
    {
        $result = false;
        $toCcBcc = [];

        /** @var Settings $settings */
        $settings = Yii::$app->settings;

        // Sender by default: No-Reply Email
        $fromEmail = static::from($settings->get("app.noreplyEmail"));
        // Sender verification
        if (empty($fromEmail)) {
            return false;
        }

        if (!empty($mailTo)) {

            $paymentConfirmationMessage = '
            <p>Assalamu Alaikum,</p>
            <p>We encountered an issue while processing your recent payment for your upcoming travel experience with Dar El Salam.</p>'
            . (!empty($paymentLink) ? '<p style="font-weight:bold;">Please use the following link to update your payment information:</p><p>' . $paymentLink . '</p>' : '') .
            '<p>May Allah make your journey blessed.</p> ';

            // Message
            $data = [
                'fieldValues' => [],
                'fieldData' => [],
                'message' => $paymentConfirmationMessage
            ];

            $subject = 'Payment Failed';

            $replyTo = '';
            $cc = '';
            $bcc = '';


            $output = Yii::$app->controller->render('@app/mail/payment-confirmation-html', $data);
            $body = Yii::$app->controller->render('@app/mail/layouts/html', ['content' => $output, 'message' => '']);

            $service = new GmailService();
            $result = $service->sendMessage($service->createMessage($fromEmail, $mailTo, $replyTo, $cc, $bcc, $subject, $body));
        }

        return $result;
    }

    /**
     * Send Abandoned Message By E-Mail
     *
     * @param Form $formModel
     * @param FormSubmission $submissionModel
     * @param array $filePaths
     * @return bool
     * @throws \Exception
     */
    public static function sendAbandonedConfirmationByEmail($formModel, $submissionData, $instance_id, $filePaths = [])
    {
//        echo unserialize($submissionData);

        $result = false;
        // Each email address should be unique between to, cc, and bcc
        $toCcBcc = [];

        /** @var Settings $settings */
        $settings = Yii::$app->settings;

//        $dataModel = $formModel->formData;
        $confirmationModel = $formModel->abandonedFormConfirmation;
//        $submissionData = $submissionModel->getSubmissionData();
//        $payments = $submissionModel->getPayments();

        // If file paths are empty, find them by model relation
//        if (empty($filePaths)) {
//            $fileModels = $submissionModel->files;
//            foreach ($fileModels as $fileModel) {
//                $filePaths[] = $fileModel->getUrl();
//            }
//        }

        // Check first: Send email must be active and Recipient is required
        if ($confirmationModel->send_email && !empty($confirmationModel->mail_to)) {
//            print_r($confirmationModel);
            // Async Email
            $async = static::async();

            // Sender by default: No-Reply Email
            $fromEmail = static::from($settings->get("app.noreplyEmail"));
            // Sender verification
            if (empty($fromEmail)) {
                return false;
            }

            // Form fields
//            $fieldsForEmail = $dataModel->getFieldsForEmail();
            // Submission data in an associative array
//            $tokens = SubmissionHelper::prepareDataForReplacementToken($submissionData, $fieldsForEmail);
            // Submission data in a multidimensional array: [0 => ['label' => '', 'value' => '']]
//            $fieldData = SubmissionHelper::prepareDataForSubmissionTable($submissionData, $fieldsForEmail);

            $mailTo = [];

            foreach ($confirmationModel->mail_to as $fieldTo) {
//                print_r($submissionData);
                // To (Get value of email field)
                $to = $submissionData[$fieldTo] ?? null;
                // Remove all illegal characters from email
                $to = filter_var($to, FILTER_SANITIZE_EMAIL);
                // Validate e-mail
                if (!filter_var($to, FILTER_VALIDATE_EMAIL) === false) {
                    if (!in_array($to, $toCcBcc)) {
                        $mailTo[] = $to;
                        $toCcBcc[] = $to;
                    }
                }
            }

            if (!empty($mailTo)) {

                // Views
                $confirmationViews = $confirmationModel->getEmailViews();

                // Message
                $data = [
                    'form_hash_id' => $formModel->getHashId(),
                    'instance_id' => $instance_id,
                    'fieldValues' => [],
                    'fieldData' => [],
//                    'fieldValues' => $tokens, // Submission data for replacement
//                    'fieldData' => $fieldData, // Submission data for print details
                    'mail_receipt_copy' => (boolean) $confirmationModel->mail_receipt_copy, // Add submission copy
                    'message' => !empty($confirmationModel->mail_message)
                        ? $confirmationModel->mail_message
                        : Yii::t('app', 'Thanks for your message'),
                ];

                // Subject
                $subject = !empty($confirmationModel->mail_subject)
                    ? $confirmationModel->mail_subject
                    : $formModel->name . ' - Confirmation';


                // Token replacement in subject
//                $subject = SubmissionHelper::replaceTokens($subject, $tokens);
//                $subject = Liquid::render($subject, $tokens);

                // Add name to From
                if (!empty($confirmationModel->mail_from_name)) {
                    if (is_array($fromEmail)) {
                        $fromEmail = [
                            key($fromEmail) => $confirmationModel->mail_from_name,
                        ];
                    } else {
                        $fromEmail = [
                            $fromEmail => $confirmationModel->mail_from_name,
                        ];
                    }
                }

                // Compose email
                /** @var Message $message */
                $message = Yii::$app->mailer->compose($confirmationViews, $data)
                    ->setFrom($fromEmail)
                    ->setTo($mailTo)
                    ->setSubject($subject);

                // Add reply to
                if (!empty($confirmationModel->mail_from)) {
                    $message->setReplyTo($confirmationModel->mail_from);
                }

                // Add cc
                if (!empty($confirmationModel->mail_cc)) {
                    $mailCc = explode(',', $confirmationModel->mail_cc);
                    $emails = [];
                    foreach ($mailCc as $email) {
                        if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
                            if (!in_array($email, $toCcBcc)) {
                                $emails[] = $email;
                                $toCcBcc[] = $email;
                            }
                        }
                    }
                    $message->setCc($emails);
                }

                // Add bcc
                if (!empty($confirmationModel->mail_bcc)) {
                    $mailBcc = explode(',', $confirmationModel->mail_bcc);
                    $emails = [];
                    foreach ($mailBcc as $email) {
                        if (!filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
                            if (!in_array($email, $toCcBcc)) {
                                $emails[] = $email;
                                $toCcBcc[] = $email;
                            }
                        }
                    }
                    $message->setBcc($emails);
                }

                // Attach files
                if ($confirmationModel->mail_attach && count($filePaths) > 0) {
                    foreach ($filePaths as $filePath) {
                        $message->attach(Yii::getAlias('@app') . DIRECTORY_SEPARATOR . $filePath);
                    }
                }

//                // Trigger Event
//                Yii::$app->trigger(SubmissionMailEvent::EVENT_NAME, new SubmissionMailEvent([
//                    'submission' => $submissionModel,
//                    'type' => SubmissionMailEvent::EVENT_TYPE_CONFIRMATION,
//                    'message' => $message,
//                    'async' => $async,
////                    'tokens' => $tokens,
//                ]));

                // Send E-mail
                if ($async) {
                    $result = $message->queue();
                } else {
                    $toEmail = implode(',',array_keys($message->getTo()));
                    $subject = $message->getSubject();

                    $replyTo = '';
                    $cc = '';
                    $bcc = '';

                    if(!empty($message->getReplyTo()) && is_array($message->getReplyTo()))
                        $replyTo = implode(',',array_keys($message->getReplyTo()));
                    if(!empty($message->getCc()) && is_array($message->getCc()))
                        $cc = implode(',',array_keys($message->getCc()));
                    if(!empty($message->getBcc()) && is_array($message->getBcc()))
                        $bcc = implode(',',array_keys($message->getBcc()));


                    Yii::$app->controller->layout = false;
                    $output = Yii::$app->controller->render('@app/mail/abandoned-html', $data);
                    $body = Yii::$app->controller->render('@app/mail/layouts/html', ['content' => $output, 'message' => '']);


                    $attachment = null;
//                    echo $submissionModel->id;
//                    print_r($payments);
//
//                    if (count($payments) > 0) {
////                        print_r($payments);
//                        $attachment = Swift_Attachment::fromPath($payments[0]->getReceiptLink() . '&pdf=1')->setFilename('Payment Receipt.pdf');
//                    }

                    $service = new GmailService();
                    $result = $service->sendMessage($service->createMessage($fromEmail, $toEmail, $replyTo, $cc, $bcc, $subject, $body, $attachment));

                }
            }
        }

        return $result;
    }

    /**
     * Return Amazon SES Regions
     * https://docs.aws.amazon.com/general/latest/gr/ses.html
     *
     * @return string[]
     */
    public static function awsSesRegions()
    {
        return [
            'us-east-2' => 'US East (Ohio)',
            'us-east-1' => 'US East (N. Virginia)',
            'us-west-1' => 'US West (N. California)',
            'us-west-2' => 'US West (Oregon)',
            'ap-south-1' => 'Asia Pacific (Mumbai)',
            'ap-northeast-2' => 'Asia Pacific (Seoul)',
            'ap-southeast-1' => 'Asia Pacific (Singapore)',
            'ap-southeast-2' => 'Asia Pacific (Sydney)',
            'ap-northeast-1' => 'Asia Pacific (Tokyo)',
            'ca-central-1' => 'Canada (Central)',
            'eu-central-1' => 'Europe (Frankfurt)',
            'eu-west-1' => 'Europe (Ireland)',
            'eu-west-2' => 'Europe (London)',
            'eu-west-3' => 'Europe (Paris)',
            'eu-north-1' => 'Europe (Stockholm)',
            'me-south-1' => 'Middle East (Bahrain)',
            'sa-east-1' => 'South America (São Paulo)',
            'us-gov-west-1' => 'AWS GovCloud (US)',
        ];
    }

}
