<?php
/**
 * Copyright (C) Baluart.COM - All Rights Reserved
 *
 * @since 1.0
 * @author Baluart E.I.R.L.
 * @copyright Copyright (c) 2015 - 2023 Baluart E.I.R.L.
 * @license http://codecanyon.net/licenses/faq Envato marketplace licenses
 * @link https://easyforms.dev/ Easy Forms
 */

namespace app\controllers;

use app\components\User;
use app\helpers\ArrayHelper;
use app\helpers\Country;
use app\models\Form;
use app\models\FormSubmission;
use app\models\search\FormSearch;
use app\modules\addons\modules\stripe\models\Stripe;
use app\modules\addons\modules\stripe\models\StripePayment;
use app\modules\addons\modules\stripe\models\StripePaymentSearch;
use app\modules\subscription\models\Subscription;
use Stripe\BalanceTransaction;
use Stripe\Payout;
use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use yii\db\Query;
use yii\helpers\Url;
use app\helpers\Html;
use yii\web\NotFoundHttpException;


/**
 * Class DashboardController
 * @package app\controllers
 */
class DashboardController extends Controller
{

    /** @inheritdoc */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    ['actions' => ['payments'], 'allow' => true, 'roles' => ['viewPaymentDashboard'], 'roleParams' => ['listing' => true]],
                    ['actions' => ['stripe-payments', 'payment-details'], 'allow' => true, 'roles' => ['viewPaymentsPage'], 'roleParams' => ['listing' => true]],
                    ['actions' => ['stripe-payouts', 'payout-info', 'get-balance-transactions'], 'allow' => true, 'roles' => ['viewPayoutsPage'], 'roleParams' => ['listing' => true]],
                    [
                        'actions' => ['index', 'account-session'],
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
        ];
    }

    /**
     * @inheritdoc
     */
    public function beforeAction($action)
    {

        if (!parent::beforeAction($action)) {
            return false;
        }

        return true; // or false to not run the action
    }

    /**
     * Dashboard
     *
     * @return mixed
     */
    public function actionIndex()
    {
        $limit = Yii::$app->params['ListGroup.listSize'] ?? 5;

        /** @var User $currentUser */
        $currentUser = Yii::$app->user;

        // Date Filters
        $startDate = Yii::$app->request->get('start_date');
        $endDate = Yii::$app->request->get('end_date');

        if ($startDate && $endDate) {
            $startDate = date("Y-m-d", strtotime(trim($startDate))) ;
            $endDate = date("Y-m-d", strtotime(trim($endDate)));
        } else {
            $startDate = date("Y-m-d", strtotime("-1 month")) ;
            $endDate = date("Y-m-d");
        }

        $users = \Da\User\Model\User::find()->where(['subscription_id' => Yii::$app->user->identity->subscription_id])->all();
        $allSubscriptionUserIds = ArrayHelper::getColumn($users, 'id');

        // Current User's filter
        $forms = $currentUser->forms()
            ->andFilterWhere(['in', '{{%form}}.created_by', $allSubscriptionUserIds])
            ->asArray()->all();
        $formIds = ArrayHelper::getColumn($forms, 'id');
        // Important restriction. If empty, don't show any form data
        $formIds = count($formIds) > 0 ? $formIds : 0;

        // Form Stats
        $filledForms = (new Query())->from('{{%stats_performance}}')
            ->select('COUNT(DISTINCT(app_id)) AS filled_forms')
            ->from('{{%stats_performance}}')
            ->where(['app_id' => $formIds])
            ->andWhere(['>', 'fills', 0])
            ->andFilterWhere(['between', 'day', $startDate, $endDate])
            ->one();

        $submittedForms = (new Query())->from('{{%stats_performance}}')
            ->select('COUNT(DISTINCT(app_id)) AS submitted_forms')
            ->from('{{%stats_performance}}')
            ->where(['app_id' => $formIds])
            ->andWhere(['>', 'conversions', 0])
            ->andFilterWhere(['between', 'day', $startDate, $endDate])
            ->one();

        $conversionTime = (new Query())->from('{{%stats_performance}}')
            ->select('SUM(conversions) AS form_conversions, SUM(conversionTime) AS form_conversion_time')
            ->from('{{%stats_performance}}')
            ->where(['app_id' => $formIds])
            ->andWhere(['<', 'conversionTime', 86400000])
            ->andFilterWhere(['between', 'day', $startDate, $endDate])
            ->one();
        $averageTime = $conversionTime['form_conversions'] > 0 ? $conversionTime['form_conversion_time'] / $conversionTime['form_conversions'] : 0;
        $averageTime = round($averageTime / 1000); // Milliseconds

        // Performance Stats
        $stats = (new Query())->from('{{%stats_performance}}')
            ->select('COUNT(DISTINCT(app_id)) AS viewed_forms, SUM(users) AS form_users, SUM(fills) AS form_starters, SUM(conversions) AS form_conversions')
            ->from('{{%stats_performance}}')
            ->where(['app_id' => $formIds])
            ->andFilterWhere(['between', 'day', $startDate, $endDate])
            ->one();

        $stats = !empty($stats) ? $stats : [];
        $stats['viewed_forms'] = $stats['viewed_forms'] ?? 0;
        $stats['filled_forms'] = $filledForms['filled_forms'] ?? 0;
        $stats['submitted_forms'] = $submittedForms['submitted_forms'] ?? 0;
        $stats['form_users'] = $stats['form_users'] ?? 0;
        $stats['form_starters'] = $stats['form_starters'] ?? 0;
        $stats['form_conversions'] = $stats['form_conversions'] ?? 0;
        $stats['form_abandons'] = !empty($stats['form_conversions']) ? $stats['form_starters'] - $stats['form_conversions'] : $stats['form_starters'];
        $stats['form_conversion_time'] = isset($conversionTime['form_conversion_time'], $conversionTime['form_conversions']) ? Yii::$app->formatter->asDuration($averageTime) : 0;
        $stats['start_typing_rate'] = !empty($stats['form_users']) ? round($stats['form_starters']  * 100 / $stats['form_users']) : 0;
        $stats['completion_rate'] =  !empty($stats['form_starters']) ? round($stats['form_conversions']  * 100 / $stats['form_starters']) : 0;
        $stats['conversion_rate'] = !empty($stats['form_users']) ? round($stats['form_conversions']  * 100 / $stats['form_users']) : 0;
        $stats['abandonment_rate'] = !empty($stats['form_users']) ? round($stats['form_abandons']  * 100 / $stats['form_users']) : 0;

        $report = (new Query())
            ->select(['s.app_id AS id', 'f.name AS name', 'SUM(s.users) AS users', 'SUM(s.fills) AS starters', 'SUM(s.conversions) AS conversions'])
            ->from('{{%stats_performance}} AS s')
            ->innerJoin('{{%form}} AS f', 's.app_id = f.id')
            ->andWhere(['s.app_id' => $formIds])
            ->andFilterWhere(['between', 'day', $startDate, $endDate])
            ->groupBy(['id'])
            ->limit($limit);

        // More users query
        $formsByUsersQuery = clone $report;
        // More starters query
        $formsByStartersQuery = clone $report;
        // More conversions query
        $formsByConversionsQuery = clone $report;

        // More users
        $formsByUsers = $formsByUsersQuery->having(['!=', 'users', 0])->orderBy('users DESC')->all();
        // More starters
        $formsByStarters = $formsByStartersQuery->having(['!=', 'starters', 0])->orderBy('starters DESC')->all();
        // More conversions
        $formsByConversions = $formsByConversionsQuery->having(['!=', 'conversions', 0])->orderBy('conversions DESC')->all();

        // Last updated forms list HERE
        $updatedForms = (new Query())
            ->select(['id', 'name', 'updated_at'])
            ->from('{{%form}} AS f')
            ->andFilterWhere(['f.id' => $formIds])
            ->orderBy('updated_at DESC')
            ->limit($limit)
            ->all();

        // Unread submissions HERE
        $unreadSubmissions = FormSubmission::find()
            ->where(['new' => 1])
            ->andFilterWhere(['form_id' => $formIds])
            ->orderBy('id DESC')
            ->limit($limit)
            ->all();

        // My Last Submitted Forms
        $mySubmissionsByForm =  (new Query())
            ->select(['fs1.id', 'fs1.form_id'])
            ->from('{{%form_submission}} AS fs1')
            ->leftJoin('{{%form_submission}} as fs2', 'fs1.form_id = fs2.form_id AND fs1.id < fs2.id')
            ->where(['fs2.id' => null])
            ->andWhere(['fs1.created_by' => Yii::$app->user->id])
            ->andFilterWhere(['fs1.form_id' => $formIds])
            ->orderBy('fs1.id DESC')
            ->limit($limit)
            ->all();

        $mySubmissionsByFormIds = ArrayHelper::getColumn($mySubmissionsByForm, 'id');

        $myLastSubmissions = FormSubmission::find()
            ->where(['id' => $mySubmissionsByFormIds])
            ->orderBy('id DESC')
            ->limit($limit)
            ->all();

        // Chart data
        $data = (new Query())
            ->select(['s.day AS labels', 'SUM(s.users) AS users', 'SUM(s.fills) AS starters', 'SUM(s.conversions) AS conversions'])
            ->from('{{%stats_performance}} AS s')
            ->andWhere(['s.app_id' => $formIds])
            ->andFilterWhere(['between', 'day', $startDate, $endDate])
            ->groupBy(['labels'])
            ->orderBy('labels ASC')
            ->all();

        // Map
        $mapData = (new Query())
            ->select(['s.geo_country AS country, COUNT(*) as submissions'])
            ->from('{{%stats_submissions}} AS s')
            ->andWhere(['s.app_id' => $formIds])
            ->andFilterWhere(['between', 's.collector_tstamp', strtotime($startDate), strtotime('+ 1 day', strtotime($endDate))])
            ->groupBy(['s.geo_country'])
            ->orderBy('s.geo_country ASC')
            ->all();

        $countries = array_flip(Country::all());
        $locations = [];
        foreach ($mapData as $m) {
            if (!empty($m['country']) && !empty($m['submissions']) && !empty($countries[$m['country']])) {
                $locations[$countries[$m['country']]] = (int) $m['submissions'];
            }
        }

        $subscription = Subscription::find()
            ->where(['user_id' => Yii::$app->user->id])
            ->andWhere(['or', ['ends_at' => null], ['>', 'ends_at', time()]])
            ->one();

//        if (!empty($subscription) && $subscription->valid() && $subscription->name == 'Basic') {
//            Yii::$app->session->setFlash(
//                'warning',
//                Yii::t('app', 'Upgrade to be able to use all of the features! ' . Html::a('Upgrade', ['subscription/user/index']))
//            );
//        }


        return $this->render('index', [
            'stats' => $stats,

            'formsByUsers' => $formsByUsers,
            'formsByStarters' => $formsByStarters,
            'formsByConversions' => $formsByConversions,

            'updatedForms' => $updatedForms,
            'unreadSubmissions' => $unreadSubmissions,
            'myLastSubmissions' => $myLastSubmissions,
            'data' => $data,
            'locations' => $locations,

            'startDate' => $startDate,
            'endDate' => $endDate,
        ]);
    }

    public function actionPayments() {

        $users = \Da\User\Model\User::find()->where(['subscription_id' => Yii::$app->user->identity->subscription_id])->all();
        $allSubscriptionUserIds = ArrayHelper::getColumn($users, 'id');

        // Date Filters
        $startDate = Yii::$app->request->get('start_date');
        $endDate = Yii::$app->request->get('end_date');

        if ($startDate && $endDate) {
            $startDate = strtotime(trim($startDate));
            $endDate = strtotime(trim($endDate));
        } else {
            $startDate = strtotime("-1 month");
            $endDate = strtotime('now');
        }

        // CHART DATA

        $query = StripePayment::find();
        $query
            ->select(["FROM_UNIXTIME(addon_stripe_payment.created_at, '%Y-%m-%d') as createdDate", 'sum(CAST(subtotal/100 AS DECIMAL(10,2))) as total'])
            ->andFilterWhere(['in', '{{%form}}.created_by', $allSubscriptionUserIds])
            ->andFilterWhere(['in', 'addon_stripe_payment.status', ['Complete','Processing']])
            ->andFilterWhere(['between', 'addon_stripe_payment.created_at', $startDate, $endDate])
            ->joinWith('form')
            ->groupBy('createdDate')
            ->asArray();

        $paymentData = ArrayHelper::map($query->all(),'createdDate', 'total');


        $query = FormSubmission::find();
        $query
            ->select(["FROM_UNIXTIME(form_submission.created_at, '%Y-%m-%d') as createdDate", 'count(form_submission.id) as numSubmissions'])
            ->andFilterWhere(['in', '{{%form}}.created_by', $allSubscriptionUserIds])
            ->andFilterWhere(['between', 'form_submission.created_at', $startDate, $endDate])
            ->joinWith('form')
            ->groupBy('createdDate')
            ->asArray();

        $submissionData = ArrayHelper::map($query->all(),'createdDate', 'numSubmissions');


        // TABLE DATA

//        $searchModel = new StripePaymentSearch();
//        $dataProvider = $searchModel->search([]);
//
//        $latestPayments = $dataProvider->query
//            ->andFilterWhere(['status' => 'Complete'])
//            ->orderBy('created_at DESC')
//            ->limit(10)
//            ->all();

        $query = StripePayment::find();
        $query
            ->select(['addon_stripe_payment.*', 'form.name as formName'])
            ->andFilterWhere(['in', 'addon_stripe_payment.status', ['Complete','Processing']])
            ->andFilterWhere(['in', '{{%form}}.created_by', $allSubscriptionUserIds])
            ->andFilterWhere(['between', 'addon_stripe_payment.created_at', $startDate, $endDate])
            ->joinWith('form')
            ->orderBy('addon_stripe_payment.created_at DESC')
            ->limit(10)
            ->asArray();

//        echo $query->createCommand()->getRawSql();

        $latestPayments = $query->all();

        $query = StripePayment::find();
        $query
            ->select(['form_id', 'form.name', 'sum(CAST(addon_stripe_payment.subtotal/100 AS DECIMAL(10,2))) as total'])
            ->andFilterWhere(['in', 'addon_stripe_payment.status', ['Complete','Processing']])
            ->andFilterWhere(['in', '{{%form}}.created_by', $allSubscriptionUserIds])
            ->joinWith('form')
            ->groupBy('form_id')
            ->asArray();

        $revenueByForm = $query->all();


//        $searchModel = new FormSearch();
//        $dataProvider = $searchModel->search([]);
//        $query = $dataProvider->query;
//        $query = Form::find();
//        $query
//            ->select(['name', 'sum(form_submission.package_total) as totalDue', 'sum(CAST(total/100 AS DECIMAL(10,2))) as totalPaid'])
//            ->andFilterWhere(['in', '{{%form}}.created_by', $allSubscriptionUserIds])
//            ->joinWith('addon_stripe_payment')
//            ->leftJoin('form_submission', 'form_submission.id = addon_stripe_payment.submission_id')
//            ->groupBy('form.id')
//            ->having(['>', 'totalDue', 0])
//            ->asArray();


        $connection = \Yii::$app->db;
        $command = $connection->createCommand('
        SELECT 
            form.name,
            sum(form_submission.package_total) AS `totalDue`,
            (select sum(total) from addon_stripe_payment where addon_stripe_payment.form_id = form_submission.form_id and status IN (\'Complete\',\'Processing\')) as `totalPaid`
            FROM `form_submission` 
            LEFT JOIN `form` ON `form`.id = form_submission.form_id
        WHERE form.created_by in (' . implode(',', $allSubscriptionUserIds) . ')
        group by form_submission.form_id
        HAVING `totalDue` > 0;
        
        ');
        $collectionRate = $command->queryAll();


//        $query = FormSubmission::find();
//        $query
//            ->select(['name', 'sum(form_submission.package_total) as totalDue', ''])
//            ->joinWith('form')
//            ->groupBy('form_id')
//            ->asArray();
//
//
//        $collectionRate = $query->all();

//        print_r($collectionRate);
//
//        echo $query->createCommand()->getRawSql();
//        die();


        return $this->render('payments', [
            'startDate' => date("Y-m-d", $startDate),
            'endDate' => date("Y-m-d", $endDate),

            'paymentData' => $paymentData,
            'submissionData' => $submissionData,

            'latestPayments' => $latestPayments,
            'revenueByForm' => $revenueByForm,
            'collectionRate' => $collectionRate,
        ]);
    }

//    public function actionExpress() {
//        $publishableKey = (Yii::$app->params['environment'] == Stripe::LIVE ? Yii::$app->params['stripeLivePublishableKey'] : Yii::$app->params['stripeTestPublishableKey']);
//        $secretKey = (Yii::$app->params['environment'] == Stripe::LIVE ? Yii::$app->params['stripeLiveSecretKey'] : Yii::$app->params['stripeTestSecretKey']);
//        $connected_account_id = Yii::$app->user->identity->stripe_account_id;
//
//        $stripe = new \Stripe\StripeClient($secretKey);
//
//        $link = $stripe->accounts->createLoginLink($connected_account_id, []);
//
//        header('Location: ' . $link->url);
//    }

    public function actionStripePayments() {

        $searchModel = new StripePaymentSearch();
        $searchModel->status = ['Complete','Processing','Failed','Requires Action','Refunded', 'Partially Refunded'];
        $dataProvider = $searchModel->search($this->request->queryParams);
//        $dataProvider->setSort([
//            'defaultOrder' => ['created'=>SORT_DESC],
//        ]);

        $dataProvider->pagination->pageSize = 20;


        return $this->render('stripepayments', [
//            'client_secret' => $this->getAccountSession(),
            'searchModel' => $searchModel,
            'dataProvider' => $dataProvider,
        ]);
    }

    public function actionStripePayouts() {

        return $this->render('stripepayouts', [
            'client_secret' => $this->getAccountSession()

        ]);
    }

    public function actionPayoutInfo()
    {
        $secretKey = (Yii::$app->params['environment'] == Stripe::LIVE ? Yii::$app->params['stripeLiveSecretKey'] : Yii::$app->params['stripeTestSecretKey']);
        $connected_account_id = Yii::$app->user->identity->stripe_account_id;

        $stripe = new \Stripe\StripeClient($secretKey, $connected_account_id);

        $selectedPayoutId = Yii::$app->request->get('payout_id');

        /** @var Payout[] $payouts */
        $payouts = $stripe->payouts->all([
            'limit' => 100,
            'expand' => ['data.destination'],
        ],['stripe_account' => $connected_account_id]);

        $formattedPayouts = [];
        foreach ($payouts as $payout) {
//            echo '<pre>' . print_r($payout, true) . '</pre>';

            $formattedPayouts[] = [
                'id' => $payout->id,
                'amount' => $payout->amount / 100,
                'currency' => strtoupper($payout->currency),
                'created' => date('Y-m-d H:i:s', $payout->created),
                'arrival_date' => date('Y-m-d H:i:s', $payout->arrival_date),
                'status' => $payout->status,
                'type' => $payout->type,
                'destination' => $payout->destination->bank_name . ' •••• ' . $payout->destination->last4
            ];
        }

        return $this->render('payout-info', [
            'client_secret' => $this->getAccountSession(),
            'payouts' => $formattedPayouts,
//            'balanceTransactions' => $balanceTransactions,
            'selectedPayoutId' => $selectedPayoutId,
        ]);
    }

    private function getAccountSession() {
        $secretKey = (Yii::$app->params['environment'] == Stripe::LIVE ? Yii::$app->params['stripeLiveSecretKey'] : Yii::$app->params['stripeTestSecretKey']);
        $connected_account_id = Yii::$app->user->identity->stripe_account_id;

        $stripe = new \Stripe\StripeClient($secretKey);

        try {

            $account_session = $stripe->accountSessions->create([
                'account' => $connected_account_id,
                'components' => [
                    'payments' => [
                        'enabled' => true,
                        'features' => [
                            'refund_management' => true,
                            'dispute_management' => true,
                            'capture_payments' => true,
                        ],
                    ],
                    'balances' => [
                        'enabled' => true,
                        'features' => [
                            'instant_payouts' => true,
                            'standard_payouts' => true,
                            'edit_payout_schedule' => true,
                        ],
                    ],
                    'account_onboarding' => [
                        'enabled' => true,
                        'features' => ['external_account_collection' => true],
                    ],
                    'account_management' => [
                        'enabled' => true,
                        'features' => ['external_account_collection' => true],
                    ],
                    'notification_banner' => [
                        'enabled' => true,
                        'features' => ['external_account_collection' => true],
                    ],
                    'documents' => ['enabled' => true],
                    'payouts' => [
                        'enabled' => true,
                        'features' => [
                            'instant_payouts' => true,
                            'standard_payouts' => true,
                            'edit_payout_schedule' => true,
                            'external_account_collection' => true,
                        ],
                    ],
                    'payouts_list' => ['enabled' => true],
                ],
            ]);


            return $account_session->client_secret;
            echo json_encode(array(
                'client_secret' => $account_session->client_secret
            ));
        } catch (Exception $e) {
            error_log("An error occurred when calling the Stripe API to create an account session: {$e->getMessage()}");
//          http_response_code(500);
            //echo json_encode(['error' => $e->getMessage()]);
            return false;
        }
    }

    public function actionGetBalanceTransactions($payout_id)
    {
        $secretKey = (Yii::$app->params['environment'] == Stripe::LIVE ? 
            Yii::$app->params['stripeLiveSecretKey'] : 
            Yii::$app->params['stripeTestSecretKey']);
        $connected_account_id = Yii::$app->user->identity->stripe_account_id;

        $stripe = new \Stripe\StripeClient($secretKey);

        $balanceTransactions = [];
        $transactions = $stripe->balanceTransactions->all([
            'payout' => $payout_id,
            'expand' => ['data.source', 'data.source.source_transfer', 'data.source.source_transfer.source_transaction', 'data.source.charge.source_transfer'],
            'limit' => 100,
        ],['stripe_account' => $connected_account_id]);

        foreach ($transactions as $transaction) {
            // Filter for only payment type and charge source
            if ($transaction->type == 'payout')
                continue;

//            if ($transaction->type !== 'payment' ||
//                !isset($transaction->source) ||
//                $transaction->source->object !== 'charge') {
//                continue;
//            }

//            echo '<pre>' . print_r($transaction, true) . '</pre>';
//            print_r($transaction->metadata);

            $paymentIntent = "";

            if (!empty($transaction) &&
                !empty($transaction->source) &&
                !empty($transaction->source->source_transfer) &&
                !empty($transaction->source->source_transfer->source_transaction))
                $paymentIntent = $transaction->source->source_transfer->source_transaction->payment_intent ?? "";

            if (empty($paymentIntent) &&
                !empty($transaction->source) &&
                !empty($transaction->source->source_transfer) &&
                !empty($transaction->source->source_transfer->metadata) &&
                !empty($transaction->source->source_transfer->metadata['payment_intent']))
                $paymentIntent = $transaction->source->source_transfer->metadata['payment_intent'];

            if (empty($paymentIntent) &&
                !empty($transaction->source) &&
                !empty($transaction->source->charge->source_transfer) &&
                !empty($transaction->source->charge->source_transfer->metadata) &&
                !empty($transaction->source->charge->source_transfer->metadata['payment_intent']))
                $paymentIntent = $transaction->source->charge->source_transfer->metadata['payment_intent'];

            $stripePayment = StripePayment::findOne(['payment_id' => $paymentIntent]);

            $balanceTransactions[] = [
                'id' => $transaction->id,
                'description' => $transaction->description,
                'type' => $transaction->type,
                'created' => date('Y-m-d H:i:s', $transaction->created),
                'status' => $transaction->status,
                'net' => $transaction->net / 100,
                'currency' => strtoupper($transaction->currency),
                'payment_intent' => $paymentIntent,
                'stripe_payment' => $stripePayment ? [
                    'id' => $stripePayment->id,
                    'form_id' => $stripePayment->form_id,
                    'form_name' => !empty($stripePayment->form) ? $stripePayment->form->name : 'Manual Payment',
                    'payment_method' => $stripePayment->payment_method,
                    'submission_id' => $stripePayment->submission_id,
                    'customer_name' => $stripePayment->customer_name,
                    'customer_email' => $stripePayment->customer_email,
                    'status' => $stripePayment->status,
                    'total' => $stripePayment->total,
                    'subtotal' => $stripePayment->subtotal,
                ] : null,
            ];
        }

        return $this->renderPartial('_balance_transactions_grid', [
            'balanceTransactions' => $balanceTransactions,
        ]);
    }

    public function actionPaymentDetails($id)
    {
        $model = StripePayment::findOne($id);
        if ($model === null) {
            throw new NotFoundHttpException('The requested payment does not exist.');
        }

        return $this->renderPartial('_payment_details', [
            'model' => $model,
        ]);
    }
}
